<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/auth_check.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';

// Check if user has permission to create sales
// Check if user has any POS-related permissions
$posPermissions = [
    'create_sales', 'process_sale', 'allow_credit_sale', 'view_products', 'view_customers'
];

if (!hasAnyPermission($posPermissions)) {
    $_SESSION['error'] = "You don't have permission to access the POS system.";
    header('Location: dashboard.php');
    exit;
}

if (false && !hasPermission('create_sales')) { // Disable the old check
    header('Location: dashboard.php');
    exit;
}

// Session timeout is now handled globally in includes/auth_check.php

$activePage = 'pos';

// Get currency from store settings
$stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'currency'");
$stmt->execute();
$currency = $stmt->fetchColumn() ?: 'Ksh';

// Get M-Pesa settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'mpesa_settings'");
$stmt->execute();
$mpesaSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
$enable_mpesa = ($mpesaSettings['enable_mpesa'] ?? '0') === '1' && 
                !empty($mpesaSettings['mpesa_consumer_key']) && 
                !empty($mpesaSettings['mpesa_consumer_secret']) && 
                !empty($mpesaSettings['mpesa_shortcode']) && 
                !empty($mpesaSettings['mpesa_passkey']) ? '1' : '0';

// Get credit system setting
$stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'enable_credit_system'");
$stmt->execute();
$enableCreditSystem = $stmt->fetchColumn() === '1';

// Get all categories
$stmt = $pdo->prepare("SELECT * FROM categories ORDER BY name");
$stmt->execute();
$categories = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no, viewport-fit=cover">
    <meta name="mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="default">
    <meta name="theme-color" content="#2563eb">
    <title>POS - Point of Sale</title>
    <?php echo getFaviconLink($pdo); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/jsqr@1.4.0/dist/jsQR.min.js"></script>
    <!-- Session Timeout Management -->
    <script src="../js/session_timeout.js"></script>
    <style>
        body { background: #f8fafc; font-family: 'Inter', sans-serif; }
        .pos-main-card {
            background: #fff;
            border-radius: 22px;
            box-shadow: 0 6px 32px rgba(37,99,235,0.07);
            padding: 2.5rem 2rem 2rem 2rem;
            margin-top: 2rem;
            margin-bottom: 2rem;
            position: relative;
            z-index: 1;
        }
        .pos-title {
            font-size: 2.3rem;
            font-weight: 800;
            color: #22223b;
            letter-spacing: 1px;
        }
        .pos-subtitle {
            color: #64748b;
            font-size: 1.1rem;
            margin-bottom: 1.5rem;
        }
        .search-bar input { border-radius: 8px; border: 1px solid #e2e8f0; }
        .search-bar input:focus { border-color: #3b82f6; box-shadow: 0 0 0 2px #3b82f633; }
        .cart-table th, .cart-table td { vertical-align: middle; }
        .cart-table th { background: #f8fafc; font-weight: 600; color: #475569; }
        .cart-table td { color: #334155; }
        .cart-table { background: #fff; border-radius: 12px; overflow: hidden; }
        .cart-summary { font-size: 1.2rem; font-weight: 700; color: #222; }
        .btn-pay { font-size: 1.1rem; font-weight: 700; border-radius: 10px; padding: 0.7rem 2.2rem; margin-right: 1rem; }
        .btn-cash { background: #2563eb; color: #fff; }
        .btn-cash:hover { background: #1d4ed8; }
        .btn-mpesa { background: #10b981; color: #fff; }
        .btn-mpesa:hover { background: #059669; }
        .btn-credit { background: #e11d48; color: #fff; }
        .btn-credit:hover { background: #be123c; }
        .btn-multi { background: #f59e0b; color: #fff; }
        .btn-multi:hover { background: #d97706; }
        .cart-actions .btn { border-radius: 8px; }
        .cart-actions .btn-danger { background: #e11d48; border: none; }
        .cart-actions .btn-danger:hover { background: #be123c; }
        
        /* Fullscreen POS Styles */
        .pos-main-card.fullscreen {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            z-index: 9999;
            margin: 0;
            border-radius: 0;
            overflow-y: auto;
            background: #fff;
        }
        
        .pos-main-card.fullscreen .pos-title {
            font-size: 2.5rem;
        }
        
        .pos-main-card.fullscreen .pos-subtitle {
            font-size: 1.2rem;
        }
        
        .pos-main-card.fullscreen .table-responsive {
            max-height: 60vh;
            overflow-y: auto;
        }
        
        .pos-main-card.fullscreen .btn-pay {
            font-size: 1.2rem;
            padding: 0.8rem 2.5rem;
        }
        
        .expand-btn {
            transition: all 0.3s ease;
            border-radius: 8px;
            padding: 0.5rem 0.75rem;
        }
        
        .expand-btn:hover {
            transform: scale(1.05);
            box-shadow: 0 4px 12px rgba(37, 99, 235, 0.2);
        }
        
        .expand-btn.fullscreen {
            background: #dc2626;
            border-color: #dc2626;
            color: white;
        }
        
        .expand-btn.fullscreen:hover {
            background: #b91c1c;
            border-color: #b91c1c;
        }
        
        /* Hide sidebar and topbar when in fullscreen */
        .pos-main-card.fullscreen ~ .sidebar,
        .pos-main-card.fullscreen ~ .topbar {
            display: none !important;
        }
        
        /* Fullscreen overlay */
        .fullscreen-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            background: rgba(0, 0, 0, 0.8);
            z-index: 9998;
            display: none;
        }
        
        .fullscreen-overlay.active {
            display: block;
        }
        
        /* Additional fullscreen improvements */
        .pos-main-card.fullscreen .container-fluid {
            padding: 2rem;
            max-width: 100%;
        }
        
        .pos-main-card.fullscreen .search-bar {
            margin-bottom: 2rem;
        }
        
        .pos-main-card.fullscreen .search-bar input {
            font-size: 1.1rem;
            padding: 0.75rem 1rem 0.75rem 3rem;
        }
        
        .pos-main-card.fullscreen .table {
            font-size: 1.1rem;
        }
        
        .pos-main-card.fullscreen .table th {
            font-size: 1.1rem;
            padding: 1rem;
        }
        
        .pos-main-card.fullscreen .table td {
            padding: 0.75rem;
        }
        
        .pos-main-card.fullscreen .cart-summary {
            font-size: 1.4rem;
        }
        
        /* Responsive adjustments for fullscreen */
        @media (max-width: 768px) {
            .pos-main-card.fullscreen {
                padding: 1rem;
            }
            
            .pos-main-card.fullscreen .pos-title {
                font-size: 2rem;
            }
            
            .pos-main-card.fullscreen .btn-pay {
                font-size: 1rem;
                padding: 0.6rem 1.5rem;
            }
        }
        .cart-actions .btn-outline-secondary { border-radius: 8px; }
        .cart-empty { color: #64748b; font-size: 1.1rem; text-align: center; padding: 2rem 0; }
        
        /* Camera button styles */
        #cameraBtn {
            transition: all 0.3s ease;
            border: 2px solid #2563eb;
            background: rgba(37, 99, 235, 0.1);
        }
        
        #cameraBtn:hover {
            background: #2563eb;
            color: white;
            transform: scale(1.05);
            box-shadow: 0 4px 12px rgba(37, 99, 235, 0.3);
        }
        
        #cameraBtn:active {
            transform: scale(0.95);
        }
        
        /* QR Scanner styles */
        .qr-scanner-container {
            background: #000;
            border: 3px solid #2563eb;
        }
        
        .scanner-frame {
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0% { box-shadow: 0 0 0 9999px rgba(0,0,0,0.5), 0 0 0 0 rgba(37, 99, 235, 0.7); }
            70% { box-shadow: 0 0 0 9999px rgba(0,0,0,0.5), 0 0 0 10px rgba(37, 99, 235, 0); }
            100% { box-shadow: 0 0 0 9999px rgba(0,0,0,0.5), 0 0 0 0 rgba(37, 99, 235, 0); }
        }
        /* Mobile Responsive Styles */
        @media (max-width: 900px) {
            .pos-main-card {
                padding: 1rem;
                margin-top: 1rem;
                margin-bottom: 1rem;
                border-radius: 15px;
            }
            
            .pos-title {
                font-size: 1.8rem;
                text-align: center;
            }
            
            .pos-subtitle {
                font-size: 1rem;
                text-align: center;
            }
            
            .search-bar {
                margin-bottom: 1rem;
            }
            
            .search-bar input {
                font-size: 1rem;
                padding: 0.75rem 1rem 0.75rem 3rem;
            }
            
            .search-bar .input-group-text {
                padding: 0.75rem;
            }
            
            .cart-table {
                font-size: 0.9rem;
            }
            
            .cart-table th,
            .cart-table td {
                padding: 0.5rem 0.25rem;
                font-size: 0.85rem;
            }
            
            .cart-summary {
                font-size: 1rem;
            }
            
            .btn-pay {
                font-size: 1rem;
                padding: 0.6rem 1.5rem;
                margin-right: 0.5rem;
                margin-bottom: 0.5rem;
            }
            
            .cart-actions .btn {
                font-size: 0.9rem;
                padding: 0.5rem 1rem;
                margin-bottom: 0.5rem;
            }
            
            /* Stack payment buttons vertically on mobile */
            .payment-buttons {
                display: flex;
                flex-direction: column;
                gap: 0.5rem;
            }
            
            .payment-buttons .btn {
                width: 100%;
                margin-right: 0;
            }
            
            /* Adjust table for mobile */
            .table-responsive {
                border-radius: 10px;
                overflow: hidden;
            }
            
            /* Mobile table optimizations */
            .cart-table {
                font-size: 0.8rem;
            }
            
            .cart-table th,
            .cart-table td {
                padding: 0.4rem 0.2rem;
                font-size: 0.75rem;
                white-space: nowrap;
                overflow: hidden;
                text-overflow: ellipsis;
                max-width: 80px;
            }
            
            /* Hide less important columns on mobile */
            .cart-table .d-none-mobile {
                display: none !important;
            }
            
            /* Make specific columns more compact on mobile */
            .cart-table th:nth-child(1), /* Product */
            .cart-table td:nth-child(1) {
                max-width: 120px;
                min-width: 100px;
            }
            
            .cart-table th:nth-child(2), /* Quantity */
            .cart-table td:nth-child(2) {
                max-width: 60px;
                min-width: 50px;
                text-align: center;
            }
            
            .cart-table th:nth-child(3), /* Price */
            .cart-table td:nth-child(3) {
                max-width: 70px;
                min-width: 60px;
                text-align: right;
            }
            
            .cart-table th:nth-child(4), /* Total */
            .cart-table td:nth-child(4) {
                max-width: 70px;
                min-width: 60px;
                text-align: right;
            }
            
            .cart-table th:nth-child(5), /* Discount */
            .cart-table td:nth-child(5) {
                max-width: 80px;
                min-width: 70px;
                text-align: center;
            }
            
            .cart-table th:nth-child(6), /* Action */
            .cart-table td:nth-child(6) {
                max-width: 60px;
                min-width: 50px;
                text-align: center;
            }
            
            /* Compact quantity controls for mobile */
            .quantity-controls {
                display: flex;
                align-items: center;
                gap: 0.25rem;
                justify-content: center;
            }
            
            .quantity-controls .btn {
                width: 28px;
                height: 28px;
                padding: 0;
                font-size: 0.8rem;
                font-weight: bold;
                display: flex;
                align-items: center;
                justify-content: center;
            }
            
            .quantity-controls .form-control {
                width: 40px;
                height: 28px;
                padding: 0.2rem;
                font-size: 0.8rem;
                text-align: center;
            }
            
            /* Make quantity controls more touch-friendly */
            .quantity-controls {
                display: flex;
                align-items: center;
                gap: 0.5rem;
            }
            
            .quantity-controls .btn {
                width: 40px;
                height: 40px;
                padding: 0;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 1.2rem;
                font-weight: bold;
            }
            
            /* Improve modal responsiveness */
            .modal-dialog {
                margin: 1rem;
                max-width: calc(100% - 2rem);
            }
            
            .modal-body {
                padding: 1rem;
            }
            
            /* QR Scanner mobile improvements */
            .qr-scanner-container {
                max-width: 100%;
                margin: 0 auto;
            }
            
            .qr-scanner-container video {
                width: 100%;
                max-width: 400px;
                height: auto;
            }
            
            .scanner-frame {
                width: 150px !important;
                height: 150px !important;
            }
            
            /* Improve touch targets */
            .btn {
                min-height: 44px;
                touch-action: manipulation;
            }
            
            /* Better spacing for mobile */
            .row {
                margin-left: -0.5rem;
                margin-right: -0.5rem;
            }
            
            .col, .col-md-6, .col-md-8, .col-md-4 {
                padding-left: 0.5rem;
                padding-right: 0.5rem;
            }
        }
        
        /* Extra small devices */
        @media (max-width: 576px) {
            .pos-main-card {
                padding: 0.75rem;
                margin-top: 0.5rem;
                margin-bottom: 0.5rem;
            }
            
            .pos-title {
                font-size: 1.5rem;
            }
            
            .pos-subtitle {
                font-size: 0.9rem;
            }
            
            .search-bar input {
                font-size: 0.9rem;
                padding: 0.6rem 0.8rem 0.6rem 2.5rem;
            }
            
            .cart-table {
                font-size: 0.8rem;
            }
            
            .cart-table th,
            .cart-table td {
                padding: 0.4rem 0.2rem;
                font-size: 0.8rem;
            }
            
            .btn-pay {
                font-size: 0.9rem;
                padding: 0.5rem 1rem;
            }
            
            .cart-actions .btn {
                font-size: 0.8rem;
                padding: 0.4rem 0.8rem;
            }
            
            /* Extra compact table for very small screens */
            .cart-table {
                font-size: 0.7rem;
            }
            
            .cart-table th,
            .cart-table td {
                padding: 0.3rem 0.1rem;
                font-size: 0.7rem;
            }
            
            /* Hide more columns on very small screens */
            .cart-table .d-none-xs {
                display: none !important;
            }
            
            /* Even more compact column widths for extra small screens */
            .cart-table th:nth-child(1), /* Product */
            .cart-table td:nth-child(1) {
                max-width: 100px;
                min-width: 80px;
            }
            
            .cart-table th:nth-child(2), /* Quantity */
            .cart-table td:nth-child(2) {
                max-width: 50px;
                min-width: 40px;
            }
            
            .cart-table th:nth-child(3), /* Price */
            .cart-table td:nth-child(3) {
                max-width: 60px;
                min-width: 50px;
            }
            
            .cart-table th:nth-child(4), /* Total */
            .cart-table td:nth-child(4) {
                max-width: 60px;
                min-width: 50px;
            }
            
            .cart-table th:nth-child(5), /* Discount */
            .cart-table td:nth-child(5) {
                max-width: 70px;
                min-width: 60px;
            }
            
            .cart-table th:nth-child(6), /* Action */
            .cart-table td:nth-child(6) {
                max-width: 50px;
                min-width: 40px;
            }
            
            /* Ultra compact quantity controls */
            .quantity-controls .btn {
                width: 24px;
                height: 24px;
                font-size: 0.7rem;
            }
            
            .quantity-controls .form-control {
                width: 35px;
                height: 24px;
                font-size: 0.7rem;
            }
            
            /* Compact action buttons */
            .cart-table .btn {
                padding: 0.2rem 0.4rem;
                font-size: 0.7rem;
                min-height: 28px;
            }
            
            /* Make scanner frame smaller */
            .scanner-frame {
                width: 120px !important;
                height: 120px !important;
            }
        }
        
        /* Landscape orientation on mobile */
        @media (max-width: 900px) and (orientation: landscape) {
            .pos-main-card {
                padding: 0.5rem;
            }
            
            .pos-title {
                font-size: 1.4rem;
            }
            
            .pos-subtitle {
                font-size: 0.9rem;
            }
            
            .table-responsive {
                max-height: 40vh;
            }
            
            .payment-buttons {
                flex-direction: row;
                flex-wrap: wrap;
            }
            
            .payment-buttons .btn {
                width: auto;
                flex: 1;
                min-width: 120px;
            }
            
            /* Better table layout in landscape */
            .cart-table th,
            .cart-table td {
                padding: 0.5rem 0.3rem;
                font-size: 0.8rem;
            }
            
            .cart-table th:nth-child(1), /* Product */
            .cart-table td:nth-child(1) {
                max-width: 150px;
                min-width: 120px;
            }
            
            .cart-table th:nth-child(2), /* Quantity */
            .cart-table td:nth-child(2) {
                max-width: 70px;
                min-width: 60px;
            }
            
            .cart-table th:nth-child(3), /* Price */
            .cart-table td:nth-child(3) {
                max-width: 80px;
                min-width: 70px;
            }
            
            .cart-table th:nth-child(4), /* Total */
            .cart-table td:nth-child(4) {
                max-width: 80px;
                min-width: 70px;
            }
            
            .cart-table th:nth-child(5), /* Discount */
            .cart-table td:nth-child(5) {
                max-width: 90px;
                min-width: 80px;
            }
            
            .cart-table th:nth-child(6), /* Action */
            .cart-table td:nth-child(6) {
                max-width: 70px;
                min-width: 60px;
            }
        }
        
        /* High DPI displays */
        @media (-webkit-min-device-pixel-ratio: 2), (min-resolution: 192dpi) {
            .btn {
                border-width: 0.5px;
            }
            
            .cart-table {
                border-width: 0.5px;
            }
        }
        
        /* Touch device optimizations */
        @media (hover: none) and (pointer: coarse) {
            .btn:hover {
                transform: none;
            }
            
            .btn:active {
                transform: scale(0.98);
            }
            
            .search-bar input:focus {
                transform: scale(1.02);
            }
        }
        
        /* Table header abbreviations for mobile */
        @media (max-width: 768px) {
            .cart-table th:nth-child(1)::after {
                content: "Product";
            }
            
            .cart-table th:nth-child(2)::after {
                content: "Qty";
            }
            
            .cart-table th:nth-child(3)::after {
                content: "Price";
            }
            
            .cart-table th:nth-child(4)::after {
                content: "Total";
            }
            
            .cart-table th:nth-child(5)::after {
                content: "Disc";
            }
            
            .cart-table th:nth-child(6)::after {
                content: "Action";
            }
            
            /* Hide original header text on mobile */
            .cart-table th {
                font-size: 0.7rem;
                font-weight: 600;
                text-align: center;
            }
            
            .cart-table th span {
                display: none;
            }
        }
        
        /* Ensure table fits in container */
        .table-responsive {
            border-radius: 10px;
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
        }
        
        /* Compact table for all mobile devices */
        @media (max-width: 576px) {
            .cart-table {
                min-width: 100%;
                table-layout: fixed;
            }
            
            .cart-table th,
            .cart-table td {
                word-wrap: break-word;
                hyphens: auto;
            }
            
            /* Product name truncation */
            .cart-table td:nth-child(1) {
                max-width: 100px;
                overflow: hidden;
                text-overflow: ellipsis;
                white-space: nowrap;
            }
            
            /* Show full product name on hover/focus */
            .cart-table td:nth-child(1):hover,
            .cart-table td:nth-child(1):focus {
                white-space: normal;
                word-wrap: break-word;
                max-width: 150px;
                position: relative;
                z-index: 10;
                background: white;
                box-shadow: 0 2px 8px rgba(0,0,0,0.1);
                border-radius: 4px;
                padding: 0.5rem;
            }
        }
            .pos-main-card { padding: 1.2rem 0.5rem; }
        }
        /* Add responsive styles */
        @media (max-width: 768px) {
            .pos-main-card {
                margin-top: 1rem;
                margin-bottom: 1rem;
                padding: 1.5rem 1rem;
                border-radius: 15px;
                width: 100%;
                margin-left: 0;
                margin-right: 0;
            }

            .container-fluid {
                padding: 0;
                margin: 0;
                width: 100%;
            }

            .row {
                margin: 0;
                width: 100%;
            }

            .col-md-10 {
                padding: 0;
                margin: 0;
                width: 100%;
            }

            /* Adjust search bars for mobile */
            .search-bar {
                flex-direction: column;
                margin: 0 0 1rem 0;
            }

            .search-bar .col-md-6 {
                width: 100%;
                padding: 0;
                margin-bottom: 0.5rem;
            }

            /* Make cart table responsive */
            .table-responsive {
                margin: 0;
                padding: 0;
            }

            .cart-table {
                font-size: 0.9rem;
                margin: 0;
            }

            .cart-table th,
            .cart-table td {
                padding: 8px;
                white-space: nowrap;
            }

            /* Adjust payment buttons */
            .d-flex.justify-content-center {
                flex-wrap: wrap;
                gap: 10px;
                margin: 1rem 0;
            }

            .btn-pay {
                width: calc(50% - 5px);
                margin: 0;
                font-size: 1rem;
                padding: 0.5rem 1rem;
            }

            /* Adjust cart summary */
            .d-flex.justify-content-between {
                flex-direction: column;
                gap: 10px;
                margin: 1rem 0;
            }

            .cart-summary {
                text-align: right;
                font-size: 1.1rem;
            }

            /* Adjust shipping input */
            #shipping {
                width: 100%;
                margin-top: 5px;
            }

            /* Adjust modals for mobile */
            .modal-dialog {
                margin: 10px;
                max-width: calc(100% - 20px);
            }

            .modal-body {
                padding: 15px;
            }

            /* Adjust customer info display */
            #selectedCustomerInfo {
                margin-top: 10px;
            }

            /* Adjust datetime display */
            #datetimeDisplay {
                font-size: 0.9rem;
                padding: 8px 16px;
                margin-bottom: 1rem;
            }
        }

        /* Additional breakpoint for very small screens */
        @media (max-width: 480px) {
            .pos-main-card {
                padding: 1rem;
                margin-top: 0.5rem;
                margin-bottom: 0.5rem;
            }

            .pos-title {
                font-size: 1.5rem;
            }

            .btn-pay {
                width: 100%;
            }

            .cart-table {
                font-size: 0.8rem;
            }

            .cart-table th,
            .cart-table td {
                padding: 6px;
            }

            #datetimeDisplay {
                font-size: 0.8rem;
                padding: 6px 12px;
            }
        }

        /* Ensure sidebar stays fixed on mobile */
        @media (max-width: 768px) {
            .sidebar {
                position: fixed;
                top: 0;
                left: 0;
                height: 100%;
                z-index: 1000;
                transform: translateX(-100%);
                transition: transform 0.3s ease-in-out;
                width: 250px;
            }

            .sidebar.show {
                transform: translateX(0);
            }

            .main-content {
                margin-left: 0;
                width: 100%;
                position: relative;
            }

            .col-md-10 {
                width: 100%;
                margin-left: 0;
                padding-left: 15px;
                padding-right: 15px;
                position: relative;
                z-index: 1;
            }

            /* Add overlay when sidebar is open */
            .sidebar-overlay {
                display: none;
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: rgba(0, 0, 0, 0.5);
                z-index: 999;
            }

            .sidebar-overlay.show {
                display: block;
            }

            /* Adjust main content when sidebar is open */
            .main-content.sidebar-open {
                margin-left: 250px;
                width: calc(100% - 250px);
            }

            /* Ensure topbar stays on top */
            .topbar {
                position: fixed;
                top: 0;
                right: 0;
                left: 0;
                z-index: 998;
                background: #fff;
            }

            /* Adjust main content padding to account for fixed topbar */
            .pos-main-card {
                margin-top: 80px;
            }
        }

        /* Additional adjustments for very small screens */
        @media (max-width: 480px) {
            .sidebar {
                width: 200px;
            }

            .main-content.sidebar-open {
                margin-left: 200px;
                width: calc(100% - 200px);
            }

            .pos-main-card {
                margin-top: 70px;
            }
        }

        .modern-stock-modal {
            border-radius: 1.5rem;
            box-shadow: 0 8px 32px rgba(37,99,235,0.18);
            background: linear-gradient(135deg, #fff1f2 0%, #f3e8ff 100%);
            border: none;
            animation: stockModalFadeIn 0.4s;
        }
        @keyframes stockModalFadeIn {
            from { transform: translateY(-30px) scale(0.95); opacity: 0; }
            to { transform: translateY(0) scale(1); opacity: 1; }
        }
        .stock-modal-icon {
            font-size: 3.5rem;
            color: #e11d48;
            animation: stockIconPop 0.5s;
        }
        @keyframes stockIconPop {
            0% { transform: scale(0.7); opacity: 0; }
            80% { transform: scale(1.15); opacity: 1; }
            100% { transform: scale(1); }
        }
        .stock-modal-message {
            font-size: 1.3rem;
            font-weight: 700;
            color: #be123c;
            letter-spacing: 0.5px;
            margin-bottom: 0.5em;
        }
        .stock-modal-list {
            margin-top: 1em;
            font-size: 1.1em;
            color: #e11d48;
            text-align: left;
            display: inline-block;
        }
        .stock-modal-list li {
            margin-bottom: 0.3em;
            font-weight: 600;
            padding-left: 1.2em;
            position: relative;
        }
        .stock-modal-list li:before {
            content: '\26A0'; /* warning sign */
            color: #e11d48;
            position: absolute;
            left: 0;
            font-size: 1em;
        }
    </style>
</head>
<body>
<div class="container-fluid py-4">
    <div class="row">
        <?php include '../includes/sidebar.php'; ?>
        <!-- Main Content -->
        <div class="col-md-10 ms-sm-auto px-md-4">
            <?php include '../includes/topbar.php'; ?>
            <div class="container-fluid">
                <div class="pos-main-card">
                    <div style="display:flex;justify-content:flex-end;margin-bottom:18px;">
                        <div id="datetimeDisplay" style="background:#f3e8ff;color:#884bf0;font-weight:600;letter-spacing:1px;font-size:1.1rem;padding:10px 24px;border-radius:12px;display:inline-block;">
                            <?php echo date('l, M d, Y - H:i:s'); ?>
                        </div>
                    </div>
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <div>
                            <div class="pos-title">Point of Sale (POS)</div>
                            <div class="pos-subtitle">Point Of Sale System</div>
                        </div>
                        <div class="d-flex align-items-center gap-2">
                            <button type="button" class="btn btn-outline-primary btn-sm expand-btn" id="expandPosBtn" title="Expand to Full Screen">
                                <i class="fas fa-expand"></i>
                            </button>
                        </div>
                    </div>
                    <div class="row search-bar mb-4 g-2">
                        <div class="col-md-6">
                            <div class="search-bar position-relative">
                                <i class="fas fa-search" style="position: absolute; left: 14px; top: 50%; transform: translateY(-50%); color: #2563eb; font-size: 1.1rem;"></i>
                                <input type="text" class="form-control ps-5" id="searchcustomers" placeholder="Search Client by Name / Email / Phone">
                                <div id="selectedCustomerInfo" class="mt-2" style="display: none;">
                                    <div class="d-flex align-items-center justify-content-between bg-light p-2 rounded">
                                        <div>
                                            <span class="fw-bold" id="selectedCustomerName"></span>
                                            <div class="text-muted small">
                                                <span id="selectedCustomerPhone"></span>
                                                <span id="selectedCustomerEmail" class="ms-2"></span>
                                            </div>
                                        </div>
                                        <button class="btn btn-sm btn-outline-danger" id="clearCustomer">
                                            <i class="fas fa-times"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="search-bar position-relative">
                                <i class="fas fa-search" style="position: absolute; left: 14px; top: 50%; transform: translateY(-50%); color: #2563eb; font-size: 1.1rem;"></i>
                                <input type="text" class="form-control ps-5" id="searchproducts" placeholder="Enter Product name / SKU / Scan bar code">
                                <button type="button" class="btn btn-outline-primary position-absolute" id="cameraBtn" style="right: 8px; top: 50%; transform: translateY(-50%); border-radius: 6px; padding: 6px 10px;" title="Scan QR Code" data-bs-toggle="tooltip" data-bs-placement="left">
                                    <i class="fas fa-camera"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                    <div class="table-responsive">
                        <table class="table table-bordered" style="table-layout: fixed; width: 100%;">
                            <thead>
                                <tr>
                                    <th style="width: 35%; text-align: left; min-width: 200px; max-width: 400px;">Product</th>
                                    <th style="width: 12%; text-align: center; min-width: 80px; max-width: 120px;">Quantity</th>
                                    <th style="width: 12%; text-align: center; min-width: 80px; max-width: 120px;">Price</th>
                                    <th style="width: 12%; text-align: center; min-width: 80px; max-width: 120px;">Total</th>
                                    <th style="width: 15%; text-align: center; min-width: 120px; max-width: 160px;">Discount (Per Item)</th>
                                    <th style="width: 7%; text-align: center; min-width: 80px; max-width: 100px;">Action</th>
                                </tr>
                            </thead>
                            <tbody id="cart-items">
                            </tbody>
                        </table>
                    </div>
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <div>
                            <span>Items: <span id="cartItems">0</span></span>
                            <span class="ms-3">Total: <span id="cartTotal">0.00</span></span>
                            <span class="ms-3">Shipping(+): <input type="number" id="shipping" value="0.00" min="0" step="0.01" style="width: 80px; border-radius: 6px; border: 1px solid #e2e8f0; padding: 2px 8px;" /></span>
                        </div>
                        <div class="cart-summary">
                            Total Payable: <span id="summaryPayable">0.00</span>
                        </div>
                    </div>
                    <div class="d-flex justify-content-center mb-2 flex-wrap">
                        <button class="btn btn-pay btn-cash" id="payCashBtn"><i class="fas fa-money-bill-wave me-2"></i>Cash</button>
                        <button class="btn btn-pay btn-mpesa" id="payMpesaBtn" style="display: <?php echo ($enable_mpesa == '1') ? 'inline-block' : 'none'; ?>;"><i class="fas fa-mobile-alt me-2"></i>Mpesa</button>
                        <button type="button" class="btn btn-pay btn-credit" id="creditSaleBtn" <?php echo !$enableCreditSystem ? 'style="display:none;"' : ''; ?> data-has-permission="<?php echo hasPermission('allow_credit_sale') ? 'true' : 'false'; ?>">Credit Sale</button>
                        <button class="btn btn-pay btn-multi" id="payMultiBtn"><i class="fas fa-layer-group me-2"></i>Multiple Pay</button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Fullscreen Overlay -->
<div class="fullscreen-overlay" id="fullscreenOverlay"></div>

<div class="modal fade" id="cashPaymentModal" tabindex="-1" aria-labelledby="cashPaymentModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content" style="border-radius: 20px; border: none; box-shadow: 0 20px 60px rgba(0,0,0,0.15);">
      <div class="modal-header" style="background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%); color: white; border-radius: 20px 20px 0 0; border: none; padding: 1.5rem;">
        <div class="d-flex align-items-center">
          <div class="me-3" style="width: 50px; height: 50px; background: rgba(255,255,255,0.2); border-radius: 50%; display: flex; align-items: center; justify-content: center;">
            <i class="fas fa-money-bill-wave" style="font-size: 1.5rem;"></i>
          </div>
          <div>
            <h5 class="modal-title mb-0" id="cashPaymentModalLabel" style="font-weight: 700; font-size: 1.3rem;">Cash Payment</h5>
            <small style="opacity: 0.9;">Enter cash amount and calculate change</small>
          </div>
        </div>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body" style="padding: 2rem;">
        <div class="row">
          <div class="col-12 mb-4">
            <div class="form-group">
              <label for="cashTendered" class="form-label" style="font-weight: 600; color: #374151; margin-bottom: 0.5rem;">
                <i class="fas fa-hand-holding-usd me-2" style="color: #2563eb;"></i>Cash Tendered
              </label>
              <div class="input-group">
                <span class="input-group-text" style="background: #f8fafc; border: 2px solid #e5e7eb; border-right: none; color: #6b7280;">
                  <i class="fas fa-coins"></i>
                </span>
                <input type="number" class="form-control" id="cashTendered" min="0" step="0.01" 
                       style="border: 2px solid #e5e7eb; border-left: none; padding: 0.75rem; font-size: 1.1rem; font-weight: 500;"
                       placeholder="Enter amount received">
              </div>
            </div>
          </div>
          
          <div class="col-12 mb-4">
            <div class="payment-summary-card" style="background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%); border-radius: 15px; padding: 1.5rem; border: 2px solid #e5e7eb;">
              <div class="row text-center">
                <div class="col-6">
                  <div class="summary-item">
                    <div class="summary-label" style="font-size: 0.9rem; color: #6b7280; font-weight: 500; margin-bottom: 0.5rem;">
                      <i class="fas fa-receipt me-1"></i>Total Due
                    </div>
                    <div class="summary-value" id="cashTotalDue" style="font-size: 1.4rem; font-weight: 700; color: #1f2937;">
                      <?php echo $currency; ?> 0.00
                    </div>
                  </div>
                </div>
                <div class="col-6">
                  <div class="summary-item">
                    <div class="summary-label" style="font-size: 0.9rem; color: #6b7280; font-weight: 500; margin-bottom: 0.5rem;">
                      <i class="fas fa-hand-holding-usd me-1"></i>Amount Paid
                    </div>
                    <div class="summary-value" id="cashAmountPaid" style="font-size: 1.4rem; font-weight: 700; color: #059669;">
                      <?php echo $currency; ?> 0.00
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          
          <div class="col-12 mb-4">
            <div class="change-display" style="background: linear-gradient(135deg, #ecfdf5 0%, #d1fae5 100%); border: 2px solid #10b981; border-radius: 15px; padding: 1.5rem; text-align: center;">
              <div class="change-label" style="font-size: 1rem; color: #065f46; font-weight: 600; margin-bottom: 0.5rem;">
                <i class="fas fa-exchange-alt me-2"></i>Change Due
              </div>
              <div class="change-value" id="cashChange" style="font-size: 2rem; font-weight: 800; color: #047857;">
                <?php echo $currency; ?> 0.00
              </div>
            </div>
          </div>
        </div>
        
        <div id="cashPaymentError" class="alert alert-danger mt-3" style="display:none; border-radius: 10px; border: none; background: #fef2f2; color: #dc2626;">
          <i class="fas fa-exclamation-circle me-2"></i>
          <span id="cashPaymentErrorText"></span>
        </div>
      </div>
      <div class="modal-footer" style="border: none; padding: 1.5rem 2rem 2rem 2rem; background: #f9fafb;">
        <button type="button" class="btn btn-light" data-bs-dismiss="modal" 
                style="border-radius: 10px; padding: 0.75rem 1.5rem; font-weight: 600; border: 2px solid #e5e7eb;">
          <i class="fas fa-times me-2"></i>Cancel
        </button>
        <button type="button" class="btn btn-primary" id="confirmCashPaymentBtn" 
                style="border-radius: 10px; padding: 0.75rem 1.5rem; font-weight: 600; background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%); border: none;">
          <i class="fas fa-check me-2"></i>Confirm Payment
        </button>
      </div>
    </div>
  </div>
</div>
<div class="modal fade" id="mpesaPaymentModal" tabindex="-1" aria-labelledby="mpesaPaymentModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="mpesaPaymentModalLabel">Mpesa Payment</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <div class="mb-3">
          <label for="mpesaAmountPaid" class="form-label">Amount</label>
          <input type="number" class="form-control" id="mpesaAmountPaid" min="0" step="0.01">
        </div>
        <div class="mb-3">
          <label for="mpesaPhone" class="form-label">Phone Number</label>
          <input type="text" class="form-control" id="mpesaPhone" placeholder="e.g. 2547XXXXXXXX">
        </div>
        <div id="mpesaPaymentStatus" class="text-success fw-bold"></div>
        <div id="mpesaPaymentError" class="text-danger mt-2" style="display:none;"></div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
        <button type="button" class="btn btn-success" id="confirmMpesaPaymentBtn">Send STK Push</button>
      </div>
    </div>
  </div>
</div>
<div class="modal fade" id="multiPayModal" tabindex="-1" aria-labelledby="multiPayModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered modal-lg">
    <div class="modal-content" style="border-radius: 20px; border: none; box-shadow: 0 20px 60px rgba(0,0,0,0.15);">
      <div class="modal-header" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); color: white; border-radius: 20px 20px 0 0; border: none; padding: 1.5rem;">
        <div class="d-flex align-items-center">
          <div class="me-3" style="width: 50px; height: 50px; background: rgba(255,255,255,0.2); border-radius: 50%; display: flex; align-items: center; justify-content: center;">
            <i class="fas fa-layer-group" style="font-size: 1.5rem;"></i>
          </div>
          <div>
            <h5 class="modal-title mb-0" id="multiPayModalLabel" style="font-weight: 700; font-size: 1.3rem;">Multiple Payment</h5>
            <small style="opacity: 0.9;">Split payment across different methods</small>
          </div>
        </div>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body" style="padding: 2rem;">
        <div class="row">
          <!-- First Payment Method -->
          <div class="col-md-6 mb-4">
            <div class="payment-method-card" style="background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%); border-radius: 15px; padding: 1.5rem; border: 2px solid #e5e7eb; height: 100%;">
              <div class="method-header mb-3" style="text-align: center;">
                <div class="method-icon" style="width: 60px; height: 60px; background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 1rem;">
                  <i class="fas fa-credit-card" style="font-size: 1.5rem; color: white;"></i>
                </div>
                <h6 style="font-weight: 700; color: #1f2937; margin-bottom: 0.5rem;">First Payment</h6>
                <small style="color: #6b7280;">Select method and amount</small>
              </div>
              
              <div class="form-group mb-3">
                <label class="form-label" style="font-weight: 600; color: #374151; margin-bottom: 0.5rem;">
                  <i class="fas fa-credit-card me-2" style="color: #3b82f6;"></i>Payment Method
                </label>
                <select class="form-select" id="multiPayMethod1" style="border: 2px solid #e5e7eb; border-radius: 10px; padding: 0.75rem; font-weight: 500;">
                  <option value="Cash">💵 Cash</option>
                  <option value="Mpesa">📱 Mpesa</option>
                  <option value="Credit" <?php echo !$enableCreditSystem ? 'style="display:none;"' : ''; ?>>💳 Credit</option>
                </select>
              </div>
              
              <div class="form-group">
                <label class="form-label" style="font-weight: 600; color: #374151; margin-bottom: 0.5rem;">
                  <i class="fas fa-coins me-2" style="color: #3b82f6;"></i>Amount
                </label>
                <div class="input-group">
                  <span class="input-group-text" style="background: #f8fafc; border: 2px solid #e5e7eb; border-right: none; color: #6b7280;">
                    <i class="fas fa-money-bill-wave"></i>
                  </span>
                  <input type="number" class="form-control" id="multiPayAmount1" min="0" step="0.01" placeholder="Enter amount"
                         style="border: 2px solid #e5e7eb; border-left: none; padding: 0.75rem; font-size: 1.1rem; font-weight: 500;">
                </div>
              </div>
            </div>
          </div>
          
          <!-- Second Payment Method -->
          <div class="col-md-6 mb-4">
            <div class="payment-method-card" style="background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%); border-radius: 15px; padding: 1.5rem; border: 2px solid #e5e7eb; height: 100%;">
              <div class="method-header mb-3" style="text-align: center;">
                <div class="method-icon" style="width: 60px; height: 60px; background: linear-gradient(135deg, #10b981 0%, #059669 100%); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 1rem;">
                  <i class="fas fa-wallet" style="font-size: 1.5rem; color: white;"></i>
                </div>
                <h6 style="font-weight: 700; color: #1f2937; margin-bottom: 0.5rem;">Second Payment</h6>
                <small style="color: #6b7280;">Select method and amount</small>
              </div>
              
              <div class="form-group mb-3">
                <label class="form-label" style="font-weight: 600; color: #374151; margin-bottom: 0.5rem;">
                  <i class="fas fa-credit-card me-2" style="color: #10b981;"></i>Payment Method
                </label>
                <select class="form-select" id="multiPayMethod2" style="border: 2px solid #e5e7eb; border-radius: 10px; padding: 0.75rem; font-weight: 500;">
                  <option value="Cash">💵 Cash</option>
                  <option value="Mpesa">📱 Mpesa</option>
                  <option value="Credit" <?php echo !$enableCreditSystem ? 'style="display:none;"' : ''; ?>>💳 Credit</option>
                </select>
              </div>
              
              <div class="form-group">
                <label class="form-label" style="font-weight: 600; color: #374151; margin-bottom: 0.5rem;">
                  <i class="fas fa-coins me-2" style="color: #10b981;"></i>Amount
                </label>
                <div class="input-group">
                  <span class="input-group-text" style="background: #f8fafc; border: 2px solid #e5e7eb; border-right: none; color: #6b7280;">
                    <i class="fas fa-money-bill-wave"></i>
                  </span>
                  <input type="number" class="form-control" id="multiPayAmount2" min="0" step="0.01" placeholder="Enter amount"
                         style="border: 2px solid #e5e7eb; border-left: none; padding: 0.75rem; font-size: 1.1rem; font-weight: 500;">
                </div>
              </div>
            </div>
          </div>
        </div>
        
        <!-- Payment Summary -->
        <div class="row mt-4">
          <div class="col-12">
            <div class="payment-summary-card" style="background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%); border: 2px solid #f59e0b; border-radius: 15px; padding: 1.5rem;">
              <div class="row text-center">
                <div class="col-md-3">
                  <div class="summary-item">
                    <div class="summary-label" style="font-size: 0.9rem; color: #92400e; font-weight: 600; margin-bottom: 0.5rem;">
                      <i class="fas fa-receipt me-1"></i>Total Due
                    </div>
                    <div class="summary-value" id="multiPayTotalDue" style="font-size: 1.3rem; font-weight: 700; color: #92400e;">
                      <?php echo $currency; ?> 0.00
                    </div>
                  </div>
                </div>
                <div class="col-md-3">
                  <div class="summary-item">
                    <div class="summary-label" style="font-size: 0.9rem; color: #92400e; font-weight: 600; margin-bottom: 0.5rem;">
                      <i class="fas fa-plus me-1"></i>Payment 1
                    </div>
                    <div class="summary-value" id="multiPayAmount1Display" style="font-size: 1.3rem; font-weight: 700; color: #3b82f6;">
                      <?php echo $currency; ?> 0.00
                    </div>
                  </div>
                </div>
                <div class="col-md-3">
                  <div class="summary-item">
                    <div class="summary-label" style="font-size: 0.9rem; color: #92400e; font-weight: 600; margin-bottom: 0.5rem;">
                      <i class="fas fa-plus me-1"></i>Payment 2
                    </div>
                    <div class="summary-value" id="multiPayAmount2Display" style="font-size: 1.3rem; font-weight: 700; color: #10b981;">
                      <?php echo $currency; ?> 0.00
                    </div>
                  </div>
                </div>
                <div class="col-md-3">
                  <div class="summary-item">
                    <div class="summary-label" style="font-size: 0.9rem; color: #92400e; font-weight: 600; margin-bottom: 0.5rem;">
                      <i class="fas fa-equals me-1"></i>Balance
                    </div>
                    <div class="summary-value" id="multiPayBalance" style="font-size: 1.3rem; font-weight: 700; color: #dc2626;">
                      <?php echo $currency; ?> 0.00
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
        
        <div id="multiPayError" class="alert alert-danger mt-3" style="display:none; border-radius: 10px; border: none; background: #fef2f2; color: #dc2626;">
          <i class="fas fa-exclamation-circle me-2"></i>
          <span id="multiPayErrorText"></span>
        </div>
      </div>
      <div class="modal-footer" style="border: none; padding: 1.5rem 2rem 2rem 2rem; background: #f9fafb;">
        <button type="button" class="btn btn-light" data-bs-dismiss="modal" 
                style="border-radius: 10px; padding: 0.75rem 1.5rem; font-weight: 600; border: 2px solid #e5e7eb;">
          <i class="fas fa-times me-2"></i>Cancel
        </button>
        <button type="button" class="btn btn-warning" id="confirmMultiPayBtn" 
                style="border-radius: 10px; padding: 0.75rem 1.5rem; font-weight: 600; background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); border: none; color: white;">
          <i class="fas fa-check me-2"></i>Confirm Payment
        </button>
      </div>
    </div>
  </div>
</div>
<div class="modal fade" id="stockWarningModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content modern-stock-modal">
      <div class="modal-body text-center">
        <div class="stock-modal-icon mb-3">
          <i class="fas fa-triangle-exclamation"></i>
        </div>
        <div id="stockWarningMessage" class="stock-modal-message">
          Cant Process sale, the following item(s) are out of stock:
        </div>
        <ul id="stockWarningList" class="stock-modal-list list-unstyled mb-0"></ul>
      </div>
    </div>
  </div>
</div>
<div class="modal fade" id="emptyCartModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content modern-stock-modal">
      <div class="modal-body text-center">
        <div class="stock-modal-icon mb-3">
          <i class="fas fa-shopping-cart"></i>
        </div>
        <div class="stock-modal-message">
          Please add items to cart first.
        </div>
      </div>
    </div>
  </div>
</div>
<div class="modal fade" id="selectCustomerModal" tabindex="-1" aria-labelledby="selectCustomerModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content modern-stock-modal">
            <div class="modal-body text-center">
                <div class="stock-modal-icon mb-3">
                    <i class="fas fa-user"></i>
                </div>
                <div class="stock-modal-message">
                    Please select a customer for credit sale.
                </div>
            </div>
        </div>
    </div>
</div>
<!-- Access Restricted Modal -->
<div class="modal fade" id="accessRestrictedModal" tabindex="-1" aria-labelledby="accessRestrictedModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content" style="border-radius: 20px; border: none; box-shadow: 0 20px 60px rgba(0,0,0,0.15);">
            <div class="modal-header" style="background: linear-gradient(135deg, #dc2626 0%, #b91c1c 100%); color: white; border-radius: 20px 20px 0 0; border: none; padding: 1.5rem;">
                <div class="d-flex align-items-center">
                    <div class="me-3" style="width: 50px; height: 50px; background: rgba(255,255,255,0.2); border-radius: 50%; display: flex; align-items: center; justify-content: center;">
                        <i class="fas fa-lock" style="font-size: 1.5rem;"></i>
                    </div>
                    <div>
                        <h5 class="modal-title mb-0" id="accessRestrictedModalLabel" style="font-weight: 700; font-size: 1.3rem;">Access Restricted</h5>
                        <small style="opacity: 0.9;">Insufficient permissions</small>
                    </div>
                </div>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" style="padding: 2rem; text-align: center;">
                <div class="mb-4">
                    <div style="width: 80px; height: 80px; background: linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%); border: 3px solid #dc2626; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 1rem;">
                        <i class="fas fa-exclamation-triangle" style="font-size: 2rem; color: #dc2626;"></i>
                    </div>
                    <h6 style="font-weight: 700; color: #1f2937; margin-bottom: 1rem;">Permission Required</h6>
                    <p style="color: #6b7280; font-size: 1rem; line-height: 1.5;">
                        You don't have permission to process credit sales. Please contact your administrator to request access to the <strong>"Allow Credit Sale"</strong> permission.
                    </p>
                </div>
                <div style="background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%); border: 2px solid #f59e0b; border-radius: 12px; padding: 1rem; margin-bottom: 1rem;">
                    <div style="font-size: 0.9rem; color: #92400e;">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Required Permission:</strong> Allow Credit Sale
                    </div>
                </div>
            </div>
            <div class="modal-footer" style="border: none; padding: 1.5rem 2rem 2rem 2rem; background: #f9fafb;">
                <button type="button" class="btn btn-primary" data-bs-dismiss="modal" 
                        style="border-radius: 10px; padding: 0.75rem 1.5rem; font-weight: 600; background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%); border: none;">
                    <i class="fas fa-check me-2"></i>I Understand
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Credit Sale Confirmation Modal -->
<div class="modal fade" id="creditSaleModal" tabindex="-1" aria-labelledby="creditSaleModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content" style="border-radius: 20px; border: none; box-shadow: 0 20px 60px rgba(0,0,0,0.15);">
            <div class="modal-header" style="background: linear-gradient(135deg, #e11d48 0%, #be123c 100%); color: white; border-radius: 20px 20px 0 0; border: none; padding: 1.2rem 1.5rem;">
                <div class="d-flex align-items-center">
                    <div class="me-3" style="width: 44px; height: 44px; background: rgba(255,255,255,0.2); border-radius: 50%; display: flex; align-items: center; justify-content: center;">
                        <i class="fas fa-credit-card" style="font-size: 1.2rem;"></i>
                    </div>
                    <div>
                        <h5 class="modal-title mb-0" id="creditSaleModalLabel" style="font-weight: 700; font-size: 1.1rem;">Confirm Credit Sale</h5>
                    </div>
                </div>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" style="padding: 1.2rem 1.5rem;">
                <div class="mb-3 text-center">
                    <div style="font-size: 1.05rem; color: #991b1b; font-weight: 600;">
                        <i class="fas fa-user me-1"></i>
                        <span id="creditCustomerName">-</span>
                    </div>
                    <div style="font-size: 0.98rem; color: #0c4a6e; font-weight: 500; margin-top: 0.5rem;">
                        <i class="fas fa-coins me-1"></i>
                        <span id="creditSaleAmount"><?php echo $currency; ?> 0.00</span>
                    </div>
                </div>
                <div class="alert alert-warning text-center" style="background: rgba(245, 158, 11, 0.1); border: 1px solid #f59e0b; border-radius: 10px; color: #92400e; font-size: 0.97rem; padding: 0.7rem 1rem;">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    This transaction will be recorded as a credit sale. The customer will be responsible for payment at a later date.
                </div>
                <div id="creditSaleError" class="alert alert-danger mt-3" style="display:none; border-radius: 10px; border: none; background: #fef2f2; color: #dc2626; font-size: 0.97rem;">
                    <i class="fas fa-exclamation-circle me-2"></i>
                    <span id="creditSaleErrorText"></span>
                </div>
            </div>
            <div class="modal-footer" style="border: none; padding: 1.2rem 1.5rem 1.5rem 1.5rem; background: #f9fafb;">
                <button type="button" class="btn btn-light" data-bs-dismiss="modal" 
                        style="border-radius: 10px; padding: 0.55rem 1.2rem; font-weight: 600; border: 2px solid #e5e7eb; font-size: 0.97rem;">
                    <i class="fas fa-times me-2"></i>Cancel
                </button>
                <button type="button" class="btn btn-danger" id="confirmCreditSale" 
                        style="border-radius: 10px; padding: 0.55rem 1.2rem; font-weight: 600; background: linear-gradient(135deg, #e11d48 0%, #be123c 100%); border: none; color: white; font-size: 0.97rem;">
                    <i class="fas fa-credit-card me-2"></i>Confirm Credit Sale
                </button>
            </div>
        </div>
    </div>
</div>
<!-- Serial Number Selection Modal -->
<div class="modal fade" id="serialNumberModal" tabindex="-1" aria-labelledby="serialNumberModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered modal-lg">
    <div class="modal-content" style="border-radius: 20px; border: none; box-shadow: 0 20px 60px rgba(0,0,0,0.15);">
      <div class="modal-header" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%); color: white; border-radius: 20px 20px 0 0; border: none; padding: 1.5rem;">
        <div class="d-flex align-items-center">
          <div class="me-3" style="width: 50px; height: 50px; background: rgba(255,255,255,0.2); border-radius: 50%; display: flex; align-items: center; justify-content: center;">
            <i class="fas fa-barcode" style="font-size: 1.5rem;"></i>
      </div>
          <div>
            <h5 class="modal-title mb-0" id="serialNumberModalLabel" style="font-weight: 700; font-size: 1.3rem;">Select Serial Numbers</h5>
            <small style="opacity: 0.9;">Choose serial numbers for this product</small>
          </div>
        </div>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body" style="padding: 2rem;">
        <form id="serialNumberForm">
          <div class="mb-4">
            <label for="serialQuantity" class="form-label" style="font-weight: 600; color: #374151;">
              <i class="fas fa-sort-numeric-up me-2" style="color: #10b981;"></i>Quantity
            </label>
            <input type="number" class="form-control" id="serialQuantity" min="1" value="1" 
                   style="border: 2px solid #e5e7eb; border-radius: 10px; padding: 0.75rem; font-size: 1.1rem;"
                   onchange="updateSerialSelection()">
          </div>
          <div class="mb-3">
            <div class="d-flex justify-content-between align-items-center mb-2">
              <label class="form-label mb-0" style="font-weight: 600; color: #374151;">
                <i class="fas fa-list-check me-2" style="color: #10b981;"></i>Available Serial Numbers
              </label>
              <div class="btn-group btn-group-sm" role="group">
                <button type="button" class="btn btn-outline-primary" onclick="selectAllSerials()" style="font-size: 0.8rem;">
                  <i class="fas fa-check-double me-1"></i>Select All
                </button>
                <button type="button" class="btn btn-outline-secondary" onclick="clearAllSerials()" style="font-size: 0.8rem;">
                  <i class="fas fa-times me-1"></i>Clear All
                </button>
              </div>
            </div>
            <div id="serialNumberList" class="border rounded p-3" style="background: #f9fafb; max-height: 300px; overflow-y: auto;"></div>
          </div>
          <div class="alert alert-info" style="border-radius: 10px; border: none; background: #eff6ff; color: #1e40af;">
            <i class="fas fa-info-circle me-2"></i>
            <strong>Note:</strong> Please select exactly the number of serial numbers matching the quantity above.
          </div>
        </form>
      </div>
      <div class="modal-footer" style="border: none; padding: 1.5rem 2rem 2rem 2rem; background: #f9fafb;">
        <button type="button" class="btn btn-light" data-bs-dismiss="modal" 
                style="border-radius: 10px; padding: 0.75rem 1.5rem; font-weight: 600; border: 2px solid #e5e7eb;">
          <i class="fas fa-times me-2"></i>Cancel
        </button>
        <button type="button" class="btn btn-primary" id="confirmSerialSelection" 
                style="border-radius: 10px; padding: 0.75rem 1.5rem; font-weight: 600; background: linear-gradient(135deg, #10b981 0%, #059669 100%); border: none;">
          <i class="fas fa-check me-2"></i>Add to Cart
        </button>
      </div>
    </div>
  </div>
</div>
<!-- Price Warning Modal -->
<div class="modal fade" id="priceWarningModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content modern-stock-modal">
      <div class="modal-body text-center">
        <div class="stock-modal-icon mb-3">
          <i class="fas fa-exclamation-triangle"></i>
        </div>
        <div id="priceWarningMessage" class="stock-modal-message">
          Price is too low! Cannot sell below cost price.
        </div>
        <div id="priceWarningDetails" class="stock-modal-list mb-3"></div>
        <button type="button" class="btn btn-primary" data-bs-dismiss="modal">
          <i class="fas fa-check me-2"></i>OK, I Understand
        </button>
      </div>
    </div>
  </div>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/js/all.min.js"></script>
<script>
    const currency = <?php echo json_encode($currency); ?>;

    // Initialize cart array
    let cart = [];
    let pendingProduct = null;

    // Function to update cart display
    function updateCartDisplay() {
        const cartBody = document.getElementById('cart-items');
        const cartItems = document.getElementById('cartItems');
        const cartTotal = document.getElementById('cartTotal');
        const summaryPayable = document.getElementById('summaryPayable');
        const shipping = parseFloat(document.getElementById('shipping').value) || 0;

        if (cart.length === 0) {
            cartBody.innerHTML = '<tr class="cart-empty"><td colspan="6">No items in cart</td></tr>';
            cartItems.textContent = '0';
            cartTotal.textContent = '0.00';
            summaryPayable.textContent = '0.00';
            return;
        }

        let total = 0;
        let totalDiscount = 0;
        cartBody.innerHTML = cart.map((item, index) => {
            const subtotal = item.price * item.quantity;
            const itemDiscount = parseFloat(item.discount) || 0;
            totalDiscount += itemDiscount;
            const itemTotal = subtotal - itemDiscount;
            total += itemTotal;
            return `
                <tr>
                    <td>
                        ${item.name}
                        ${item.serial_numbers && item.serial_numbers.length > 0 ? 
                            `<br><small class=\"text-muted\">Serial: ${item.serial_numbers.join(', ')}</small>` 
                            : ''}
                    </td>
                    <td style="text-align: center;">
                        <input type=\"number\" class=\"form-control form-control-sm\" 
                               value=\"${item.quantity}\" min=\"1\" 
                               onchange=\"updateQuantity(${index}, this.value)\">
                    </td>
                    <td style="text-align: center;">
                        <input type=\"number\" class=\"form-control form-control-sm\" 
                               value=\"${item.price}\" min=\"${(item.cost_price !== undefined ? item.cost_price : 0).toFixed(2)}\" step=\"0.01\" 
                               onchange=\"updatePrice(${index}, this.value)\"
                               oninput=\"validatePriceInput(this, ${index}, ${item.cost_price !== undefined ? item.cost_price : 0})\">
                    </td>
                    <td style="text-align: center;">${itemTotal.toFixed(2)}</td>
                    <td style="text-align: center;">
                        <input type=\"number\" class=\"form-control form-control-sm\" 
                               value=\"${item.discount ? (item.discount / item.quantity).toFixed(2) : 0}\" min=\"0\" 
                               onchange=\"updateDiscount(${index}, this.value)\">

                    </td>
                    <td style="text-align: center;">
                        <button class=\"btn btn-sm btn-danger\" onclick=\"removeItem(${index})\">
                            <i class=\"fas fa-trash\"></i>
                        </button>
                    </td>
                </tr>
            `;
        }).join('');

        // Update cart items count to reflect total quantity
        const totalItems = cart.reduce((sum, item) => sum + item.quantity, 0);
        cartItems.textContent = totalItems;
        cartTotal.textContent = total.toFixed(2);
        summaryPayable.textContent = (total + shipping).toFixed(2);
    }

    // Function to update item quantity
    function updateQuantity(index, newValue) {
        const newQuantity = parseInt(newValue, 10);
        if (newQuantity > 0 && newQuantity <= cart[index].stock_quantity) {
            cart[index].quantity = newQuantity;
            updateCartDisplay();
        }
    }

    // Function to update item price with validation
    function updatePrice(index, newValue) {
        const newPrice = parseFloat(newValue);
        const item = cart[index];
        const minPrice = item.cost_price !== undefined ? item.cost_price : 0; // Cost price as minimum with safety check
        
        // Debug: Log the update values
        console.log('Price update:', {
            index: index,
            newPrice: newPrice,
            minPrice: minPrice,
            item: item,
            cart: cart
        });
        
        if (newPrice < minPrice) {
            // Show price warning modal
            document.getElementById('priceWarningMessage').textContent = 
                `Price is too low! Cannot sell below cost price.`;
            document.getElementById('priceWarningDetails').innerHTML = 
                `<li>Product: ${item.name}</li>
                 <li>Cost Price: ${currency} ${minPrice.toFixed(2)}</li>
                 <li>Minimum Allowed: ${currency} ${minPrice.toFixed(2)}</li>
                 <li>Attempted Price: ${currency} ${newPrice.toFixed(2)}</li>`;
            
            const modal = new bootstrap.Modal(document.getElementById('priceWarningModal'));
            modal.show();
            
            // Reset the input to the current price
            const priceInputs = document.querySelectorAll('input[onchange*="updatePrice"]');
            if (priceInputs[index]) {
                priceInputs[index].value = item.price;
                priceInputs[index].style.borderColor = '';
                priceInputs[index].style.backgroundColor = '';
            }
            return;
        }
        
        if (newPrice >= 0) {
            cart[index].price = newPrice;
            updateCartDisplay();
        }
    }

    // Function to update item discount
    function updateDiscount(index, discount) {
        const item = cart[index];
        const discountPerItem = parseFloat(discount) || 0;
        const totalDiscount = discountPerItem * item.quantity; // Multiply by quantity
        
        // Calculate maximum allowed discount (price - cost price) * quantity
        const costPrice = item.cost_price || 0;
        const maxDiscountPerItem = Math.max(0, (item.price ?? 0) - costPrice);
        const maxTotalDiscount = maxDiscountPerItem * item.quantity;
        
        const newDiscount = Math.min(Math.max(0, totalDiscount), maxTotalDiscount);
        item.discount = newDiscount;
        
        // Show warning if discount exceeds maximum
        if (totalDiscount > maxTotalDiscount) {
            // Show discount warning modal
            document.getElementById('priceWarningMessage').textContent = 
                `Discount is too high! Cannot discount below cost price.`;
            document.getElementById('priceWarningDetails').innerHTML = 
                `<li>Product: ${item.name}</li>
                 <li>Cost Price: ${currency} ${costPrice.toFixed(2)}</li>
                 <li>Current Price: ${currency} ${item.price.toFixed(2)}</li>
                 <li>Maximum Discount Per Item: ${currency} ${maxDiscountPerItem.toFixed(2)}</li>
                 <li>Maximum Total Discount: ${currency} ${maxTotalDiscount.toFixed(2)}</li>
                 <li>Attempted Discount: ${currency} ${totalDiscount.toFixed(2)}</li>`;
            
            const modal = new bootstrap.Modal(document.getElementById('priceWarningModal'));
            modal.show();
        }
        
        // Add visual feedback to discount input
        const discountInput = document.querySelector(`input[onchange*="updateDiscount(${index}"]`);
        if (discountInput) {
            if (totalDiscount > maxTotalDiscount) {
                discountInput.style.borderColor = '#dc3545';
                discountInput.style.backgroundColor = '#fff5f5';
            } else {
                discountInput.style.borderColor = '';
                discountInput.style.backgroundColor = '';
            }
        }
        
        updateCartDisplay();
    }

    // Function to remove item from cart
    function removeFromCart(index) {
        cart.splice(index, 1);
        updateCartDisplay();
    }

    // Customer search functionality
    const customerSearchInput = document.getElementById('searchcustomers');
    const selectedCustomerInfo = document.getElementById('selectedCustomerInfo');
    const selectedCustomerName = document.getElementById('selectedCustomerName');
    const selectedCustomerPhone = document.getElementById('selectedCustomerPhone');
    const selectedCustomerEmail = document.getElementById('selectedCustomerEmail');
    const clearCustomerBtn = document.getElementById('clearCustomer');
    let selectedCustomer = null;

    // Create suggestions container for customers
    const customerSuggestionsContainer = document.createElement('div');
    customerSuggestionsContainer.className = 'search-suggestions';
    customerSuggestionsContainer.style.cssText = 'position: absolute; top: 100%; left: 0; right: 0; background: white; border: 1px solid #e2e8f0; border-radius: 8px; box-shadow: 0 4px 16px rgba(37,99,235,0.08); z-index: 1000; display: none; max-height: 300px; overflow-y: auto;';
    customerSearchInput.parentNode.appendChild(customerSuggestionsContainer);

    // Customer search event listener
    customerSearchInput.addEventListener('input', function() {
        const term = this.value.trim();
        if (term.length < 2) {
            customerSuggestionsContainer.style.display = 'none';
            return;
        }

        fetch(`../api/search_customers.php?term=${encodeURIComponent(term)}`)
            .then(response => response.json())
            .then(data => {
                if (data.success && data.customers.length > 0) {
                    customerSuggestionsContainer.innerHTML = data.customers.map(customer => `
                        <div class="search-suggestion-item" 
                             style="padding: 10px 15px; cursor: pointer; border-bottom: 1px solid #e2e8f0; transition: all 0.2s ease;" 
                             data-id="${customer.id}" 
                             data-name="${customer.name}" 
                             data-phone="${customer.phone || ''}" 
                             data-email="${customer.email || ''}"
                             data-credit-limit="${customer.credit_limit || 0}"
                             onmouseover="this.style.backgroundColor='#f8fafc'"
                             onmouseout="this.style.backgroundColor='white'">
                            <div style="font-weight: 500;">${customer.name}</div>
                            <div style="font-size: 0.9em; color: #64748b;">
                                ${customer.phone ? `📞 ${customer.phone}` : ''} 
                                ${customer.email ? `📧 ${customer.email}` : ''}
                            </div>
                        </div>
                    `).join('');
                    customerSuggestionsContainer.style.display = 'block';
                } else {
                    // Show "Customer not found, Click to Add Customer" message when no customers are found
                    customerSuggestionsContainer.innerHTML = `
                        <div class="add-customer-suggestion" 
                             style="padding: 15px; text-align: center; color: #dc2626; font-weight: bold; cursor: pointer; transition: all 0.2s ease;"
                             onmouseover="this.style.backgroundColor='#fef2f2'"
                             onmouseout="this.style.backgroundColor='white'"
                             onclick="openAddCustomerModal()">
                            Customer not found, Click to Add Customer
                        </div>
                    `;
                    customerSuggestionsContainer.style.display = 'block';
                }
            })
            .catch(error => {
                console.error('Error searching customers:', error);
                customerSuggestionsContainer.style.display = 'none';
            });
    });

    // Function to open Add Customer modal
    // Global variable for credit system status
    const creditSystemEnabled = <?php echo $enableCreditSystem ? 'true' : 'false'; ?>;
    
    function openAddCustomerModal() {
        // Create modal if it doesn't exist
        if (!document.getElementById('addCustomerModal')) {
            const currencySymbol = '<?php echo $currency; ?>';
            
            let creditLimitField = '';
            if (creditSystemEnabled) {
                creditLimitField = `
                    <div class="mb-3">
                        <label class="form-label">Credit Limit <span class="text-danger">*</span></label>
                        <div class="input-group">
                            <span class="input-group-text">${currencySymbol}</span>
                            <input type="number" class="form-control" name="credit_limit" 
                                   value="0" 
                                   min="0" max="10000" step="0.01" required>
                        </div>
                        <div class="invalid-feedback">Please enter a valid credit limit</div>
                        <small class="text-muted">Maximum credit limit is ${currencySymbol} 10,000.00</small>
                    </div>
                `;
            }
            
            const modalHTML = `
                <div class="modal fade" id="addCustomerModal" tabindex="-1">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title">Add New Customer</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <form method="post" id="addCustomerForm" class="needs-validation" novalidate>
                                    <div class="mb-3">
                                        <label class="form-label">Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" name="name" required>
                                        <div class="invalid-feedback">Please enter customer name</div>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">Phone <span class="text-danger">*</span></label>
                                        <input type="tel" class="form-control" name="phone" required pattern="[0-9]{10,}">
                                        <div class="invalid-feedback">Please enter a valid phone number</div>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">Email <span class="text-danger">*</span></label>
                                        <input type="email" class="form-control" name="email" required>
                                        <div class="invalid-feedback">Please enter a valid email address</div>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">Address <span class="text-danger">*</span></label>
                                        <textarea class="form-control" name="address" rows="2" required></textarea>
                                        <div class="invalid-feedback">Please enter customer address</div>
                                    </div>
                                    ${creditLimitField}
                                    <div class="text-end">
                                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                        <button type="submit" name="add_customer" class="btn btn-primary">Add Customer</button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            document.body.insertAdjacentHTML('beforeend', modalHTML);
            
            // Add form submission handler
            document.getElementById('addCustomerForm').addEventListener('submit', function(e) {
                e.preventDefault();
                
                const formData = new FormData(this);
                formData.append('add_customer', '1');
                
                // Show loading state
                const submitBtn = this.querySelector('button[type="submit"]');
                const originalText = submitBtn.textContent;
                submitBtn.textContent = 'Adding...';
                submitBtn.disabled = true;
                
                fetch('../pages/customers.php', {
                    method: 'POST',
                    body: formData,
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest'
                    }
                })
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP error! status: ${response.status}`);
                    }
                    return response.json();
                })
                .then(data => {
                    if (data.success) {
                        // Close modal and show success message
                        const modal = bootstrap.Modal.getInstance(document.getElementById('addCustomerModal'));
                        modal.hide();
                        alert('Customer added successfully!');
                        
                        // Get the form data to create the customer object
                        const formData = new FormData(document.getElementById('addCustomerForm'));
                        const newCustomer = {
                            id: data.data ? data.data.id : null, // If the API returns the new customer ID
                            name: formData.get('name'),
                            phone: formData.get('phone'),
                            email: formData.get('email'),
                            credit_limit: parseFloat(formData.get('credit_limit') || 0)
                        };
                        
                        // Automatically select the newly added customer
                        selectedCustomer = newCustomer;
                        
                        // Update UI to show selected customer
                        selectedCustomerName.textContent = selectedCustomer.name;
                        selectedCustomerPhone.textContent = selectedCustomer.phone ? `Phone: ${selectedCustomer.phone}` : '';
                        selectedCustomerEmail.textContent = selectedCustomer.email ? `Email: ${selectedCustomer.email}` : '';
                        selectedCustomerInfo.style.display = 'block';
                        
                        // Clear the search input and hide suggestions
                        customerSearchInput.value = '';
                        customerSuggestionsContainer.style.display = 'none';
                        
                        // Clear the form
                        document.getElementById('addCustomerForm').reset();
                    } else {
                        alert('Error: ' + (data.message || 'Unknown error occurred'));
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error adding customer: ' + error.message);
                })
                .finally(() => {
                    // Reset button state
                    submitBtn.textContent = originalText;
                    submitBtn.disabled = false;
                });
            });
        }
        
        // Show the modal
        const modal = new bootstrap.Modal(document.getElementById('addCustomerModal'));
        modal.show();
    }

    // Handle customer selection
    customerSuggestionsContainer.addEventListener('click', function(e) {
        const item = e.target.closest('.search-suggestion-item');
        if (item) {
            selectedCustomer = {
                id: item.dataset.id,
                name: item.dataset.name,
                phone: item.dataset.phone,
                email: item.dataset.email,
                credit_limit: parseFloat(item.dataset.creditLimit || item.dataset.credit || 0)
            };
            
            // Update UI
            selectedCustomerName.textContent = selectedCustomer.name;
            selectedCustomerPhone.textContent = selectedCustomer.phone ? `Phone: ${selectedCustomer.phone}` : '';
            selectedCustomerEmail.textContent = selectedCustomer.email ? `Email: ${selectedCustomer.email}` : '';
            selectedCustomerInfo.style.display = 'block';
            customerSearchInput.value = '';
            customerSuggestionsContainer.style.display = 'none';
        }
    });

    // Clear selected customer
    clearCustomerBtn.addEventListener('click', function() {
        selectedCustomer = null;
        selectedCustomerInfo.style.display = 'none';
        customerSearchInput.value = '';
    });

    // Product search functionality
    const productSearchInput = document.getElementById('searchproducts');
    const productSuggestionsContainer = document.createElement('div');
    productSuggestionsContainer.className = 'search-suggestions';
    productSuggestionsContainer.style.cssText = 'position: absolute; top: 100%; left: 0; right: 0; background: white; border: 1px solid #e2e8f0; border-radius: 8px; box-shadow: 0 4px 16px rgba(37,99,235,0.08); z-index: 1000; display: none; max-height: 300px; overflow-y: auto;';
    productSearchInput.parentNode.appendChild(productSuggestionsContainer);

    // Function to handle barcode scan
    async function handleBarcodeScan(code) {
        console.log('handleBarcodeScan called with code:', code);
        try {
            const url = `../api/lookup_product.php?code=${encodeURIComponent(code)}`;
            console.log('Fetching from URL:', url);
            
            const response = await fetch(url);
            const data = await response.json();
            console.log('API response:', data);
            
            if (data.success && data.product) {
                const product = {
                    id: data.product.id,
                    name: data.product.name,
                    price: parseFloat(data.product.price),
                    cost_price: typeof data.product.cost_price !== 'undefined' ? parseFloat(data.product.cost_price) : 0,
                    stock_quantity: parseInt(data.product.quantity),
                    quantity: 1,
                    discount: 0,
                    require_serials: parseInt(data.product.require_serials || "0"),
                    original_price: parseFloat(data.product.price)
                };

                // Check if price is zero
                if (product.price === 0) {
                    document.getElementById('stockWarningMessage').textContent = 
                        'Cannot initiate sale. Prices in stock management are not set. Please set prices first.';
                    const list = document.getElementById('stockWarningList');
                    list.innerHTML = `<li>${product.name}</li>`;
                    const modal = new bootstrap.Modal(document.getElementById('stockWarningModal'));
                    modal.show();
                    setTimeout(() => {
                        modal.hide();
                    }, 3000);
                    return;
                }

                // Check if product is out of stock
                if (product.stock_quantity <= 0) {
                    document.getElementById('stockWarningMessage').textContent = 
                        'Cannot add item. The following item is out of stock:';
                    const list = document.getElementById('stockWarningList');
                    list.innerHTML = `<li>${product.name}</li>`;
                    const modal = new bootstrap.Modal(document.getElementById('stockWarningModal'));
                    modal.show();
                    setTimeout(() => {
                        modal.hide();
                    }, 2000);
                    return;
                }

                // Check if product requires serials but has no stock (which might indicate no serials)
                if (product.require_serials === 1 && product.stock_quantity === 0) {
                    showNotification('This product requires serial numbers but has no stock. Please add serial numbers to inventory first.', 'warning');
                    return;
                }

                if (product.require_serials === 1) {
                    console.log('Product requires serial numbers, checking if entered code is a specific serial...');
                    
                    // First, check if the entered code is a specific serial number for this product
                    const serialCheckUrl = `check_specific_serial.php?product_id=${product.id}&serial_number=${encodeURIComponent(code)}`;
                    console.log('Checking specific serial:', serialCheckUrl);
                    
                    try {
                        const serialCheckResponse = await fetch(serialCheckUrl);
                        const serialCheckData = await serialCheckResponse.json();
                        console.log('Specific serial check response:', serialCheckData);
                        
                        if (serialCheckData.success) {
                            if (serialCheckData.available) {
                            // The entered code is an available serial number - add directly to cart
                            console.log('Serial number found and available, adding directly to cart');
                            
                            const existingIndex = cart.findIndex(item => item.id === product.id);
                            if (existingIndex > -1) {
                                // Check if this specific serial is already in cart
                                const existingSerials = cart[existingIndex].serial_numbers || [];
                                if (existingSerials.includes(code)) {
                                    showNotification('This serial number is already in the cart!', 'warning');
                                    return;
                                }
                                
                                // Add the serial to existing cart item
                                if (!cart[existingIndex].serial_numbers) {
                                    cart[existingIndex].serial_numbers = [];
                                }
                                cart[existingIndex].serial_numbers.push(code);
                                cart[existingIndex].quantity = cart[existingIndex].serial_numbers.length;
                            } else {
                                // Create new cart item with this specific serial
                                cart.push({
                                    id: product.id,
                                    name: product.name,
                                    price: parseFloat(product.price),
                                    cost_price: parseFloat(product.cost_price || 0),
                                    original_price: parseFloat(product.price),
                                    quantity: 1,
                                    discount: 0,
                                    stock_quantity: product.stock_quantity,
                                    require_serials: product.require_serials,
                                    serial_numbers: [code]
                                });
                            }
                            
                            updateCartDisplay();
                            showNotification(`Added ${product.name} with serial number ${code} to cart!`, 'success');
                            return;
                            } else {
                                // Serial number exists but is not available
                                showNotification(`Serial number ${code} is not available: ${serialCheckData.message}`, 'warning');
                                return;
                            }
                        }
                                            } catch (error) {
                            console.error('Error checking specific serial:', error);
                            // Continue with normal serial number modal if check fails
                            showNotification('Could not verify specific serial number, showing all available serials...', 'info');
                        }
                    
                    // If not a specific serial or check failed, show the serial selection modal
                    console.log('Not a specific serial, fetching all available serials...');
                    const serialUrl = `get_available_serials.php?product_id=${product.id}`;
                    console.log('Fetching serials from:', serialUrl);
                    
                    const serialResponse = await fetch(serialUrl);
                    const serialData = await serialResponse.json();
                    console.log('Serial data response:', serialData);
                    
                    if (!serialData.success) {
                        console.error('Serial fetch error:', serialData.message);
                        showNotification('Error: ' + (serialData.message || 'Could not fetch serial numbers'), 'error');
                        return;
                    }
                    const serialNumbers = Array.isArray(serialData.serial_numbers) ? serialData.serial_numbers : [];
                    console.log('Available serial numbers:', serialNumbers);
                    
                    if (serialNumbers.length === 0) {
                        // No serial numbers available for this product
                        showNotification('This product requires serial numbers but none are available. Please add serial numbers to inventory first.', 'warning');
                        return;
                    }
                    
                    pendingProduct = product;
                    showSerialNumberModal(serialNumbers, product);
                } else {
                    // Add to cart as usual
                    const existingIndex = cart.findIndex(item => item.id === product.id);
                    if (existingIndex > -1) {
                        if (cart[existingIndex].quantity + 1 <= product.stock_quantity) {
                            cart[existingIndex].quantity += 1;
                        } else {
                            alert('Cannot add more items. Maximum stock quantity reached.');
                            return;
                        }
                    } else {
                        cart.push({
                            id: product.id,
                            name: product.name,
                            price: parseFloat(product.price),
                            cost_price: parseFloat(product.cost_price || 0),
                            original_price: parseFloat(product.price),
                            quantity: 1,
                            discount: 0,
                            stock_quantity: product.stock_quantity,
                            require_serials: product.require_serials
                        });
                    }
                    updateCartDisplay();
                    // Close the product suggestions popup
                    productSuggestionsContainer.style.display = 'none';
                }
            } else {
                // Product not found
                alert('Product not found');
            }
        } catch (error) {
            console.error('Error processing barcode:', error);
            alert('Error processing barcode');
        }
    }

    // Barcode scan event listener
    let barcodeBuffer = '';
    let lastKeyTime = 0;
    const BARCODE_TIMEOUT = 50; // milliseconds between keypresses to consider it a barcode scan

    productSearchInput.addEventListener('keydown', function(e) {
        const currentTime = new Date().getTime();
        
        // If it's been too long since the last keypress, reset the buffer
        if (currentTime - lastKeyTime > BARCODE_TIMEOUT) {
            barcodeBuffer = '';
        }
        
        lastKeyTime = currentTime;

        // If Enter key is pressed
        if (e.key === 'Enter') {
            e.preventDefault(); // Prevent form submission
            if (barcodeBuffer.length > 0) {
                handleBarcodeScan(barcodeBuffer);
                barcodeBuffer = ''; // Reset buffer
                this.value = ''; // Clear input
            }
        } else {
            // Add character to buffer
            barcodeBuffer += e.key;
        }
    });

    // Product search event listener (for manual search)
    productSearchInput.addEventListener('input', function() {
        const term = this.value.trim();
        if (term.length < 2) {
            productSuggestionsContainer.style.display = 'none';
            return;
        }

        fetch(`../api/search_products.php?term=${encodeURIComponent(term)}`)
            .then(response => response.json())
            .then(data => {
                if (data.success && data.products.length > 0) {
                    productSuggestionsContainer.innerHTML = data.products.map(product => `
                        <div class="search-suggestion-item" 
                             style="padding: 10px 15px; cursor: pointer; border-bottom: 1px solid #e2e8f0; transition: all 0.2s ease;" 
                             data-id="${product.id}" 
                             data-name="${product.name}" 
                             data-price="${product.selling_price}" 
                             data-cost-price="${product.cost_price}" 
                             data-stock="${product.stock_quantity}"
                             data-require-serials="${product.require_serials}"
                             onmouseover="this.style.backgroundColor='#f8fafc'"
                             onmouseout="this.style.backgroundColor='white'">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <div style="font-weight: 500;">${product.name}</div>
                                    <div style="font-size: 0.9em; color: #64748b;">
                                        Stock: ${product.stock_quantity}
                                    </div>
                                </div>
                                <div class="text-primary fw-bold">
                                    ${currency} ${parseFloat(product.selling_price).toFixed(2)}
                                </div>
                            </div>
                        </div>
                    `).join('');
                    productSuggestionsContainer.style.display = 'block';
                } else {
                    productSuggestionsContainer.style.display = 'none';
                }
            })
            .catch(error => {
                console.error('Error searching products:', error);
                productSuggestionsContainer.style.display = 'none';
            });
    });

    // Handle product selection from suggestions
    productSuggestionsContainer.addEventListener('click', function(e) {
        const item = e.target.closest('.search-suggestion-item');
        if (item) {
            const product = {
                id: item.dataset.id,
                name: item.dataset.name,
                price: parseFloat(item.dataset.price),
                cost_price: parseFloat(item.dataset.costPrice),
                stock_quantity: parseInt(item.dataset.stock),
                quantity: 1,
                discount: 0,
                require_serials: parseInt(item.dataset.requireSerials || "0"),
                original_price: parseFloat(item.dataset.price)
            };

            // Check if price is zero
            if (product.price === 0) {
                document.getElementById('stockWarningMessage').textContent = 
                    'Cannot initiate sale. Prices in stock management are not set. Please set prices first.';
                const list = document.getElementById('stockWarningList');
                list.innerHTML = `<li>${product.name}</li>`;
                const modal = new bootstrap.Modal(document.getElementById('stockWarningModal'));
                modal.show();
                setTimeout(() => {
                    modal.hide();
                }, 3000);
                return;
            }

            // Check if product is out of stock
            if (product.stock_quantity <= 0) {
                document.getElementById('stockWarningMessage').textContent = 
                    'Cannot add item. The following item is out of stock:';
                const list = document.getElementById('stockWarningList');
                list.innerHTML = `<li>${product.name}</li>`;
                const modal = new bootstrap.Modal(document.getElementById('stockWarningModal'));
                modal.show();
                setTimeout(() => {
                    modal.hide();
                }, 2000);
                return;
            }

            if (product.require_serials === 1) {
                // Fetch available serial numbers before showing modal
                fetch(`get_available_serials.php?product_id=${product.id}`)
                    .then(response => response.json())
                    .then(data => {
                        if (!data.success) {
                            alert('Error: ' + (data.message || 'Could not fetch serial numbers'));
                            return;
                        }
                        const serialNumbers = Array.isArray(data.serial_numbers) ? data.serial_numbers : [];
                        pendingProduct = product;
                        showSerialNumberModal(serialNumbers, product);
                    })
                    .catch(error => {
                        console.error('Error fetching serial numbers:', error);
                        alert('Error fetching serial numbers');
                    });
            } else {
                // Add to cart as usual
                const existingIndex = cart.findIndex(item => item.id === product.id);
                if (existingIndex > -1) {
                    if (cart[existingIndex].quantity + 1 <= product.stock_quantity) {
                        cart[existingIndex].quantity += 1;
                    } else {
                        alert('Cannot add more items. Maximum stock quantity reached.');
                        return;
                    }
                } else {
                    cart.push({
                        id: product.id,
                        name: product.name,
                        price: parseFloat(product.price),
                        cost_price: typeof product.cost_price !== 'undefined' ? parseFloat(product.cost_price) : 0,
                        original_price: parseFloat(product.price),
                        quantity: 1,
                        discount: 0,
                        stock_quantity: product.stock_quantity,
                        require_serials: product.require_serials
                    });
                }
                updateCartDisplay();
                // Close the product suggestions popup
                productSuggestionsContainer.style.display = 'none';
            }
        }
    });

    // Close suggestions when clicking outside
    document.addEventListener('click', function(e) {
        if (!customerSearchInput.contains(e.target) && !customerSuggestionsContainer.contains(e.target)) {
            customerSuggestionsContainer.style.display = 'none';
        }
        if (!productSearchInput.contains(e.target) && !productSuggestionsContainer.contains(e.target)) {
            productSuggestionsContainer.style.display = 'none';
        }
    });

    // Add live update for shipping input
    document.getElementById('shipping').addEventListener('input', updateCartDisplay);

    // Cash Payment Modal Logic
    const payCashBtn = document.getElementById('payCashBtn');
    const cashPaymentModal = new bootstrap.Modal(document.getElementById('cashPaymentModal'));
    const cashAmountPaidInput = document.getElementById('cashAmountPaid');
    const cashTenderedInput = document.getElementById('cashTendered');
    const cashChangeInput = document.getElementById('cashChange');
    const cashPaymentError = document.getElementById('cashPaymentError');
    const confirmCashPaymentBtn = document.getElementById('confirmCashPaymentBtn');

    function checkCartStock() {
        let outOfStockItems = [];
        for (let item of cart) {
            if (item.quantity > item.stock_quantity || item.stock_quantity < 1) {
                outOfStockItems.push(item.name);
            }
        }
        if (outOfStockItems.length > 0) {
            showStockWarning(outOfStockItems);
            return false;
        }
        return true;
    }

    function showStockWarning(itemNames) {
        document.getElementById('stockWarningMessage').textContent =
            'Cant Process sale, the following item(s) are out of stock:';
        const list = document.getElementById('stockWarningList');
        list.innerHTML = '';
        itemNames.forEach(name => {
            const li = document.createElement('li');
            li.textContent = name;
            list.appendChild(li);
        });
        const modal = new bootstrap.Modal(document.getElementById('stockWarningModal'));
        modal.show();
        setTimeout(() => {
            modal.hide();
        }, 2000);
    }

    payCashBtn.addEventListener('click', function() {
        if (cart.length === 0) {
            const modal = new bootstrap.Modal(document.getElementById('emptyCartModal'));
            modal.show();
            setTimeout(() => {
                modal.hide();
            }, 2000);
            return;
        }
        if (!checkCartStock()) return;
        const totalPayable = document.getElementById('summaryPayable').textContent;
        document.getElementById('cashTotalDue').textContent = totalPayable;
        document.getElementById('cashAmountPaid').textContent = totalPayable;
        document.getElementById('cashTendered').value = '';
        document.getElementById('cashChange').textContent = currency + ' 0.00';
        document.getElementById('cashPaymentError').style.display = 'none';
        cashPaymentModal.show();
    });

    document.getElementById('cashTendered').addEventListener('input', function() {
        const totalPayable = parseFloat(document.getElementById('cashTotalDue').textContent.replace(currency, '').trim());
        const tendered = parseFloat(this.value) || 0;
        const change = Math.max(0, tendered - totalPayable);
        document.getElementById('cashChange').textContent = currency + ' ' + change.toFixed(2);
        
        if (tendered < totalPayable) {
            document.getElementById('cashPaymentErrorText').textContent = `Insufficient amount. Remaining: ${currency} ${(totalPayable - tendered).toFixed(2)}`;
            document.getElementById('cashPaymentError').style.display = 'block';
        } else {
            document.getElementById('cashPaymentError').style.display = 'none';
        }
    });

    confirmCashPaymentBtn.addEventListener('click', function() {
        const totalPayable = parseFloat(document.getElementById('cashTotalDue').textContent.replace(currency, '').trim());
        const tendered = parseFloat(document.getElementById('cashTendered').value) || 0;
        const change = parseFloat(document.getElementById('cashChange').textContent.replace(currency, '').trim()) || 0;
        
        if (tendered <= 0) {
            document.getElementById('cashPaymentErrorText').textContent = 'Please enter a valid amount.';
            document.getElementById('cashPaymentError').style.display = 'block';
            return;
        }
        
        if (tendered < totalPayable) {
            document.getElementById('cashPaymentErrorText').textContent = 'Insufficient amount.';
            document.getElementById('cashPaymentError').style.display = 'block';
            return;
        }

        // Prepare sale data
        const saleData = {
            cart: cart,
            customer: selectedCustomer,
            shipping: parseFloat(document.getElementById('shipping').value) || 0,
            total: totalPayable,
            paid: totalPayable,
            cash_tendered: tendered,
            change_amount: change,
            payment_method: 'Cash',
            balance: 0
        };

        // Send to backend for processing
        fetch('../api/process_sale.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(saleData)
        })
        .then(response => {
            if (!response.ok) {
                return response.text().then(text => {
                    throw new Error(`Server error: ${response.status} ${response.statusText}\n${text}`);
                });
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                // Print receipt (open in new tab or popup)
                window.open(data.receipt_url, '_blank');
                // Reset cart and UI
                cart = [];
                updateCartDisplay();
                cashPaymentModal.hide();
            } else {
                const errorMessage = data.message || 'Payment failed. Please try again.';
                const debugInfo = data.debug ? `\nDebug info: ${data.debug}` : '';
                document.getElementById('cashPaymentErrorText').textContent = errorMessage + debugInfo;
                document.getElementById('cashPaymentError').style.display = 'block';
                console.error('Payment error:', data);
            }
        })
        .catch(error => {
            console.error('Payment processing error:', error);
            document.getElementById('cashPaymentErrorText').textContent = 'Error processing payment: ' + error.message;
            document.getElementById('cashPaymentError').style.display = 'block';
        });
    });

    // Mpesa Payment Modal Logic
    const payMpesaBtn = document.getElementById('payMpesaBtn');
    const mpesaPaymentModal = new bootstrap.Modal(document.getElementById('mpesaPaymentModal'));
    const mpesaAmountPaidInput = document.getElementById('mpesaAmountPaid');
    const mpesaPhoneInput = document.getElementById('mpesaPhone');
    const mpesaPaymentStatus = document.getElementById('mpesaPaymentStatus');
    const mpesaPaymentError = document.getElementById('mpesaPaymentError');
    const confirmMpesaPaymentBtn = document.getElementById('confirmMpesaPaymentBtn');

    payMpesaBtn.addEventListener('click', function() {
        if (cart.length === 0) {
            const modal = new bootstrap.Modal(document.getElementById('emptyCartModal'));
            modal.show();
            setTimeout(() => {
                modal.hide();
            }, 2000);
            return;
        }
        if (!checkCartStock()) return;
        mpesaAmountPaidInput.value = document.getElementById('summaryPayable').textContent;
        mpesaPhoneInput.value = selectedCustomer && selectedCustomer.phone ? selectedCustomer.phone : '';
        mpesaPaymentStatus.textContent = '';
        mpesaPaymentError.style.display = 'none';
        mpesaPaymentModal.show();
    });

    confirmMpesaPaymentBtn.addEventListener('click', function() {
        const totalPayable = parseFloat(document.getElementById('summaryPayable').textContent);
        const paid = parseFloat(mpesaAmountPaidInput.value) || 0;
        const phone = mpesaPhoneInput.value.trim();
        
        if (paid <= 0 || !phone) {
            mpesaPaymentError.textContent = 'Please enter a valid amount and phone number.';
            mpesaPaymentError.style.display = 'block';
            return;
        }

        // Validate phone number format
        if (!/^(?:254|\+254|0)?([7-9]{1}[0-9]{8})$/.test(phone)) {
            mpesaPaymentError.textContent = 'Please enter a valid M-Pesa phone number.';
            mpesaPaymentError.style.display = 'block';
            return;
        }

        // Format phone number to 254 format
        const formattedPhone = phone.replace(/^(?:254|\+254|0)/, '254');

        mpesaPaymentStatus.textContent = 'Sending STK Push...';
        mpesaPaymentError.style.display = 'none';
        confirmMpesaPaymentBtn.disabled = true;

        // Call backend to trigger STK Push
        fetch('../api/mpesa_stkpush.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ 
                amount: paid, 
                phone: formattedPhone,
                customer_id: selectedCustomer ? selectedCustomer.id : null,
                customer_name: selectedCustomer ? selectedCustomer.name : null
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                mpesaPaymentStatus.textContent = 'STK Push sent. Awaiting payment confirmation...';
                // After a short delay, process the sale as Mpesa
                setTimeout(() => {
                    const saleData = {
                        cart: cart,
                        customer: selectedCustomer,
                        shipping: parseFloat(document.getElementById('shipping').value) || 0,
                        total: totalPayable,
                        paid: paid,
                        payment_method: 'Mpesa',
                        balance: 0
                    };
                    fetch('../api/process_sale.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify(saleData)
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            window.open(data.receipt_url, '_blank');
                            cart = [];
                            updateCartDisplay();
                            mpesaPaymentModal.hide();
                        } else {
                            mpesaPaymentError.textContent = data.message || 'Payment failed.';
                            mpesaPaymentError.style.display = 'block';
                        }
                    })
                    .catch(error => {
                        mpesaPaymentError.textContent = 'Error processing payment.';
                        mpesaPaymentError.style.display = 'block';
                    })
                    .finally(() => {
                        confirmMpesaPaymentBtn.disabled = false;
                    });
                }, 2000);
            } else {
                mpesaPaymentError.textContent = data.message || 'STK Push failed.';
                mpesaPaymentError.style.display = 'block';
                mpesaPaymentStatus.textContent = '';
                confirmMpesaPaymentBtn.disabled = false;
            }
        })
        .catch(error => {
            mpesaPaymentError.textContent = 'Error initiating M-Pesa payment.';
            mpesaPaymentError.style.display = 'block';
            mpesaPaymentStatus.textContent = '';
            confirmMpesaPaymentBtn.disabled = false;
        });
    });

    // Function to show credit error modals
    function showCreditErrorModal(creditData) {
        let modalTitle = 'Credit Sale Not Allowed';
        let modalIcon = 'fas fa-exclamation-triangle';
        let modalIconColor = '#dc2626';
        let modalMessage = '';
        
        if (creditData.error === 'Customer is suspended and cannot make credit purchases') {
            modalTitle = 'Customer Suspended';
            modalIcon = 'fas fa-user-slash';
            modalMessage = `<strong>${creditData.customer_name}</strong> is currently suspended and cannot make credit purchases.`;
        } else if (creditData.error === 'Customer does not have credit enabled') {
            modalTitle = 'Credit Not Enabled';
            modalIcon = 'fas fa-credit-card';
            modalMessage = `<strong>${creditData.customer_name}</strong> does not have credit enabled.<br>Credit Limit: <strong>${currency} ${creditData.credit_limit.toFixed(2)}</strong>`;
        } else if (creditData.error === 'Sale amount exceeds available credit') {
            modalTitle = 'Insufficient Credit';
            modalIcon = 'fas fa-triangle-exclamation';
            modalMessage = `
                <strong>${creditData.customer_name}</strong> has insufficient credit for this sale.
                <div class="mt-3">
                    <div class="row text-start">
                        <div class="col-6"><strong>Credit Limit:</strong></div>
                        <div class="col-6">${currency} ${creditData.credit_limit.toFixed(2)}</div>
                        <div class="col-6"><strong>Current Usage:</strong></div>
                        <div class="col-6">${currency} ${creditData.current_usage.toFixed(2)}</div>
                        <div class="col-6"><strong>Available Credit:</strong></div>
                        <div class="col-6 text-success">${currency} ${creditData.available_credit.toFixed(2)}</div>
                        <div class="col-6"><strong>Sale Amount:</strong></div>
                        <div class="col-6 text-danger">${currency} ${creditData.sale_amount.toFixed(2)}</div>
                        <div class="col-6"><strong>Shortfall:</strong></div>
                        <div class="col-6 text-danger">${currency} ${creditData.deficit.toFixed(2)}</div>
                    </div>
                </div>
            `;
        }
        
        // Create and show the modal
        const modalHtml = document.createElement('div');
        modalHtml.innerHTML = `
            <div class="modal fade" id="creditErrorModal" tabindex="-1" aria-labelledby="creditErrorModalLabel" aria-hidden="true">
                <div class="modal-dialog modal-dialog-centered">
                    <div class="modal-content" style="border: none; border-radius: 15px; box-shadow: 0 10px 30px rgba(0,0,0,0.2);">
                        <div class="modal-body text-center p-4">
                            <div class="mb-3">
                                <i class="${modalIcon}" style="font-size: 3rem; color: ${modalIconColor};"></i>
                            </div>
                            <h5 class="modal-title mb-3" style="font-weight: 700; color: #1f2937;">${modalTitle}</h5>
                            <div class="stock-modal-message" style="color: #6b7280; font-size: 0.95rem;">
                                ${modalMessage}
                            </div>
                            <div class="mt-4">
                                <button type="button" class="btn btn-primary" data-bs-dismiss="modal" style="border-radius: 10px; padding: 8px 24px;">
                                    <i class="fas fa-check me-2"></i>Understood
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        document.body.appendChild(modalHtml);
        const creditErrorModal = new bootstrap.Modal(document.getElementById('creditErrorModal'));
        creditErrorModal.show();
        
        // Clean up modal after it's hidden
        creditErrorModal._element.addEventListener('hidden.bs.modal', function() {
            document.getElementById('creditErrorModal').remove();
        });
    }

    // Credit Sale Button Logic
    const payCreditBtn = document.getElementById('creditSaleBtn');
    payCreditBtn.addEventListener('click', function() {
        // Check permission first
        const hasPermission = payCreditBtn.getAttribute('data-has-permission') === 'true';
        if (!hasPermission) {
            const accessRestrictedModal = new bootstrap.Modal(document.getElementById('accessRestrictedModal'));
            accessRestrictedModal.show();
            return;
        }
        
        if (cart.length === 0) {
            const modal = new bootstrap.Modal(document.getElementById('emptyCartModal'));
            modal.show();
            setTimeout(() => {
                modal.hide();
            }, 2000);
            return;
        }

        // Check for items with zero or negative quantity
        const outOfStockItems = cart.filter(item => item.quantity <= 0);
        if (outOfStockItems.length > 0) {
            const stockWarningModal = new bootstrap.Modal(document.getElementById('stockWarningModal'));
            const stockWarningList = document.getElementById('stockWarningList');
            stockWarningList.innerHTML = outOfStockItems.map(item => 
                `<li class="text-danger">${item.name} (Current Stock: ${item.stock})</li>`
            ).join('');
            stockWarningModal.show();
            return;
        }

        if (!selectedCustomer) {
            const modal = new bootstrap.Modal(document.getElementById('selectCustomerModal'));
            modal.show();
            setTimeout(() => {
                modal.hide();
            }, 2000);
            return;
        }

        // Check customer credit using comprehensive validation
        const totalPayable = parseFloat(document.getElementById('summaryPayable').textContent);
        
        fetch(`../api/check_customer_credit.php?customer_id=${selectedCustomer.id}&sale_amount=${totalPayable}`)
            .then(response => response.json())
            .then(data => {
                if (!data.success) {
                    // Credit check failed - show appropriate error modal
                    showCreditErrorModal(data);
                    return;
                }
                
                // Credit check passed - show confirmation modal with warning if applicable
                document.getElementById('creditCustomerName').textContent = data.customer_name;
                document.getElementById('creditSaleAmount').textContent = currency + ' ' + totalPayable.toFixed(2);
                
                // Show warning if credit usage will be high after this sale
                const creditSaleError = document.getElementById('creditSaleError');
                const creditSaleErrorText = document.getElementById('creditSaleErrorText');
                
                if (data.warning_level !== 'none') {
                    let warningMessage = '';
                    let warningClass = 'alert-warning';
                    
                    if (data.warning_level === 'critical') {
                        warningMessage = `⚠️ Warning: This sale will bring ${data.customer_name}'s credit usage to ${data.usage_percent_after_sale.toFixed(1)}% of their limit.`;
                        warningClass = 'alert-danger';
                    } else if (data.warning_level === 'warning') {
                        warningMessage = `⚠️ Notice: This sale will bring ${data.customer_name}'s credit usage to ${data.usage_percent_after_sale.toFixed(1)}% of their limit.`;
                        warningClass = 'alert-warning';
                    } else if (data.warning_level === 'moderate') {
                        warningMessage = `ℹ️ Info: After this sale, ${data.customer_name} will have ${currency} ${data.available_after_sale.toFixed(2)} credit remaining.`;
                        warningClass = 'alert-info';
                    }
                    
                    creditSaleError.className = 'alert mt-3 ' + warningClass;
                    creditSaleError.style = 'border-radius: 10px; border: none; font-size: 0.97rem;';
                    creditSaleErrorText.innerHTML = warningMessage;
                    creditSaleError.style.display = 'block';
                } else {
                    creditSaleError.style.display = 'none';
                }
                
                const creditSaleModal = new bootstrap.Modal(document.getElementById('creditSaleModal'));
                creditSaleModal.show();
            })
            .catch(error => {
                console.error('Credit check error:', error);
                alert('Could not verify customer credit. Please try again.');
            });
    });

    // Handle credit sale confirmation
    document.getElementById('confirmCreditSale').addEventListener('click', function() {
        const totalPayable = parseFloat(document.getElementById('summaryPayable').textContent);
        const saleData = {
            cart: cart,
            customer: selectedCustomer,
            shipping: parseFloat(document.getElementById('shipping').value) || 0,
            total: totalPayable,
            paid: totalPayable, // Fix: set paid to totalPayable for credit sale
            payment_method: 'credit',
            balance: 0,
            payments: [
                {
                    method: 'Credit',
                    amount: totalPayable
                }
            ]
        };

        fetch('../api/process_sale.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(saleData)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                window.open(data.receipt_url, '_blank');
                cart = [];
                updateCartDisplay();
                // Hide the modal
                const creditSaleModal = bootstrap.Modal.getInstance(document.getElementById('creditSaleModal'));
                creditSaleModal.hide();
            } else {
                alert(data.message || 'Credit sale failed.');
            }
        })
        .catch(() => alert('Error processing credit sale.'));
    });

    // Multiple Pay Modal Logic
    const payMultiBtn = document.getElementById('payMultiBtn');
    const multiPayModal = new bootstrap.Modal(document.getElementById('multiPayModal'));
    const multiPayMethod1 = document.getElementById('multiPayMethod1');
    const multiPayAmount1 = document.getElementById('multiPayAmount1');
    const multiPayMethod2 = document.getElementById('multiPayMethod2');
    const multiPayAmount2 = document.getElementById('multiPayAmount2');
    const multiPayError = document.getElementById('multiPayError');
    const confirmMultiPayBtn = document.getElementById('confirmMultiPayBtn');

    payMultiBtn.addEventListener('click', function() {
        if (cart.length === 0) {
            const modal = new bootstrap.Modal(document.getElementById('emptyCartModal'));
            modal.show();
            setTimeout(() => {
                modal.hide();
            }, 2000);
            return;
        }
        if (!checkCartStock()) return;
        const totalPayable = document.getElementById('summaryPayable').textContent;
        document.getElementById('multiPayTotalDue').textContent = totalPayable;
        document.getElementById('multiPayAmount1Display').textContent = currency + ' 0.00';
        document.getElementById('multiPayAmount2Display').textContent = currency + ' 0.00';
        document.getElementById('multiPayBalance').textContent = totalPayable;
        multiPayAmount1.value = '';
        multiPayAmount2.value = '';
        document.getElementById('multiPayError').style.display = 'none';
        
        // Handle credit options based on system settings
        const creditOption1 = multiPayMethod1.querySelector('option[value="Credit"]');
        const creditOption2 = multiPayMethod2.querySelector('option[value="Credit"]');
        
        if (creditOption1) {
            if (!creditSystemEnabled) {
                creditOption1.style.display = 'none';
                creditOption1.disabled = true;
            } else {
                creditOption1.style.display = 'block';
                creditOption1.disabled = false;
            }
        }
        
        if (creditOption2) {
            if (!creditSystemEnabled) {
                creditOption2.style.display = 'none';
                creditOption2.disabled = true;
            } else {
                creditOption2.style.display = 'block';
                creditOption2.disabled = false;
            }
        }
        
        // Reset to default values (avoid Credit if disabled)
        multiPayMethod1.value = 'Cash';
        multiPayMethod2.value = 'Mpesa';
        
        multiPayModal.show();
    });

    // Add real-time calculation for multiple payment
    multiPayAmount1.addEventListener('input', updateMultiPaySummary);
    multiPayAmount2.addEventListener('input', updateMultiPaySummary);
    
    // Add listeners for payment method changes to show credit warnings
    multiPayMethod1.addEventListener('change', updateMultiPaySummary);
    multiPayMethod2.addEventListener('change', updateMultiPaySummary);

    function updateMultiPaySummary() {
        const totalPayable = parseFloat(document.getElementById('multiPayTotalDue').textContent.replace(currency, '').trim());
        const amount1 = parseFloat(multiPayAmount1.value) || 0;
        const amount2 = parseFloat(multiPayAmount2.value) || 0;
        const totalPaid = amount1 + amount2;
        const balance = totalPayable - totalPaid;
        const method1 = multiPayMethod1.value;
        const method2 = multiPayMethod2.value;

        document.getElementById('multiPayAmount1Display').textContent = currency + ' ' + amount1.toFixed(2);
        document.getElementById('multiPayAmount2Display').textContent = currency + ' ' + amount2.toFixed(2);
        document.getElementById('multiPayBalance').textContent = currency + ' ' + balance.toFixed(2);

        // Update balance color based on amount
        const balanceElement = document.getElementById('multiPayBalance');
        if (balance < 0) {
            balanceElement.style.color = '#dc2626'; // Red for overpayment
        } else if (balance > 0) {
            balanceElement.style.color = '#dc2626'; // Red for underpayment
        } else {
            balanceElement.style.color = '#059669'; // Green for exact payment
        }
        
        // Show credit warnings
        const multiPayError = document.getElementById('multiPayError');
        const multiPayErrorText = document.getElementById('multiPayErrorText');
        const usingCredit = method1 === 'Credit' || method2 === 'Credit';
        const creditAmount = method1 === 'Credit' ? amount1 : (method2 === 'Credit' ? amount2 : 0);
        
        if (usingCredit && creditAmount > 0) {
            if (!selectedCustomer) {
                multiPayErrorText.innerHTML = '<i class="fas fa-user-plus me-2"></i>Please select a customer to use credit payment.';
                multiPayError.className = 'alert alert-warning mt-3';
                multiPayError.style.display = 'block';
            } else {
                multiPayErrorText.innerHTML = `<i class="fas fa-credit-card me-2"></i>Credit payment of ${currency} ${creditAmount.toFixed(2)} will be charged to <strong>${selectedCustomer.name}</strong>.`;
                multiPayError.className = 'alert alert-info mt-3';
                multiPayError.style.display = 'block';
            }
        } else {
            multiPayError.style.display = 'none';
        }
    }

    confirmMultiPayBtn.addEventListener('click', function() {
        const totalPayable = parseFloat(document.getElementById('multiPayTotalDue').textContent.replace(currency, '').trim());
        const method1 = multiPayMethod1.value;
        const amount1 = parseFloat(multiPayAmount1.value) || 0;
        const method2 = multiPayMethod2.value;
        const amount2 = parseFloat(multiPayAmount2.value) || 0;
        
        // Check permission for credit payment
        const hasPermission = payCreditBtn.getAttribute('data-has-permission') === 'true';
        if ((method1 === 'Credit' || method2 === 'Credit') && !hasPermission) {
            multiPayModal.hide();
            const accessRestrictedModal = new bootstrap.Modal(document.getElementById('accessRestrictedModal'));
            accessRestrictedModal.show();
            return;
        }
        
        // Basic validation
        if ((amount1 + amount2).toFixed(2) != totalPayable.toFixed(2)) {
            document.getElementById('multiPayErrorText').textContent = 'The sum of both payments must equal the total payable.';
            document.getElementById('multiPayError').style.display = 'block';
            return;
        }
        if (method1 === method2) {
            document.getElementById('multiPayErrorText').textContent = 'Please select two different payment methods.';
            document.getElementById('multiPayError').style.display = 'block';
            return;
        }
        
        // Check if credit is being used and validate
        const usingCredit = method1 === 'Credit' || method2 === 'Credit';
        const creditAmount = method1 === 'Credit' ? amount1 : (method2 === 'Credit' ? amount2 : 0);
        
        if (usingCredit) {
            // Check if customer is selected
            if (!selectedCustomer) {
                document.getElementById('multiPayErrorText').textContent = 'Please select a customer before using credit payment.';
                document.getElementById('multiPayError').style.display = 'block';
                return;
            }
            
            // Validate credit availability
            fetch(`../api/check_customer_credit.php?customer_id=${selectedCustomer.id}&sale_amount=${creditAmount}`)
                .then(response => response.json())
                .then(creditData => {
                    if (!creditData.success) {
                        // Credit validation failed
                        let errorMessage = '';
                        if (creditData.error === 'Customer is suspended and cannot make credit purchases') {
                            errorMessage = `${creditData.customer_name} is suspended and cannot use credit.`;
                        } else if (creditData.error === 'Customer does not have credit enabled') {
                            errorMessage = `${creditData.customer_name} does not have credit enabled.`;
                        } else if (creditData.error === 'Sale amount exceeds available credit') {
                            errorMessage = `Credit amount (${currency} ${creditAmount.toFixed(2)}) exceeds available credit (${currency} ${creditData.available_credit.toFixed(2)}) for ${creditData.customer_name}.`;
                        } else {
                            errorMessage = creditData.error || 'Credit validation failed.';
                        }
                        
                        document.getElementById('multiPayErrorText').textContent = errorMessage;
                        document.getElementById('multiPayError').style.display = 'block';
                        return;
                    }
                    
                    // Credit validation passed - proceed with payment processing
                    document.getElementById('multiPayError').style.display = 'none';
                    processMultiplePayment(method1, amount1, method2, amount2, totalPayable);
                })
                .catch(error => {
                    console.error('Credit validation error:', error);
                    document.getElementById('multiPayErrorText').textContent = 'Could not verify customer credit. Please try again.';
                    document.getElementById('multiPayError').style.display = 'block';
                });
        } else {
            // No credit involved - proceed directly
            document.getElementById('multiPayError').style.display = 'none';
            processMultiplePayment(method1, amount1, method2, amount2, totalPayable);
        }
    });
    
    // Function to process multiple payment after all validations
    function processMultiplePayment(method1, amount1, method2, amount2, totalPayable) {
        
        // For Mpesa, trigger STK Push for the relevant amount
        const doMpesa = (method1 === 'Mpesa' && amount1 > 0) || (method2 === 'Mpesa' && amount2 > 0);
        const mpesaAmount = method1 === 'Mpesa' ? amount1 : (method2 === 'Mpesa' ? amount2 : 0);
        const mpesaStep = () => {
            if (!doMpesa) return Promise.resolve();
            let phone = selectedCustomer && selectedCustomer.phone ? selectedCustomer.phone : prompt('Enter phone number for Mpesa payment:');
            if (!phone) return Promise.reject('Mpesa phone number required.');
            return fetch('../api/mpesa_stkpush.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ amount: mpesaAmount, phone: phone })
            })
            .then(response => response.json())
            .then(data => {
                if (!data.success) throw new Error(data.message || 'Mpesa payment failed.');
            });
        };
        mpesaStep().then(() => {
            // Prepare sale data
            const payments = [
                { method: method1, amount: amount1 },
                { method: method2, amount: amount2 }
            ];
            const saleData = {
                cart: cart,
                customer: selectedCustomer,
                shipping: parseFloat(document.getElementById('shipping').value) || 0,
                total: totalPayable,
                payments: payments
            };
            fetch('../api/process_sale.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(saleData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    window.open(data.receipt_url, '_blank');
                    cart = [];
                    updateCartDisplay();
                    multiPayModal.hide();
                } else {
                    document.getElementById('multiPayErrorText').textContent = data.message || 'Multiple payment failed.';
                    document.getElementById('multiPayError').style.display = 'block';
                }
            })
            .catch(() => {
                document.getElementById('multiPayErrorText').textContent = 'Error processing multiple payment.';
                document.getElementById('multiPayError').style.display = 'block';
            });
        }).catch(err => {
            document.getElementById('multiPayErrorText').textContent = err.message || err;
            document.getElementById('multiPayError').style.display = 'block';
        });
    }

    // Live updating clock
    function updateDateTime() {
        const now = new Date();
        const options = { weekday: 'long', year: 'numeric', month: 'short', day: '2-digit', hour: '2-digit', minute: '2-digit', second: '2-digit' };
        document.getElementById('datetimeDisplay').textContent = now.toLocaleString('en-US', options);
    }
    setInterval(updateDateTime, 1000);
    updateDateTime();

    function addToCart(product) {
        const existingItem = cart.find(item => item.id === product.id);
        
        if (existingItem) {
            existingItem.quantity++;
            existingItem.total = existingItem.price * existingItem.quantity;
            updateCart();
        } else {
            // Fetch available serial numbers for the product
            fetch(`get_available_serials.php?product_id=${product.id}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const serialNumbers = data.serial_numbers || [];
                        cart.push({
                            id: product.id,
                            name: product.name,
                            price: parseFloat(product.price),
                            cost_price: parseFloat(product.cost_price || 0),
                            original_price: parseFloat(product.price),
                            quantity: 1,
                            discount: 0,
                            total: parseFloat(product.price), // Calculate total for new item
                            serial_numbers: serialNumbers, // Add serial numbers
                            stock_quantity: product.stock_quantity,
                            require_serials: product.require_serials
                        });
                        updateCart();
                    } else {
                        alert('Error fetching serial numbers: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error fetching serial numbers');
                });
        }
    }

    function updateCart() {
        const cartBody = document.getElementById('cart-items');
        cartBody.innerHTML = '';
        let total = 0;

        // Ensure all items have proper totals calculated
        cart.forEach((item, index) => {
            if (!item.total) {
                item.total = item.price * item.quantity;
            }
        });

        cart.forEach((item, index) => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td>
                    ${item.name}
                    ${item.serial_numbers && item.serial_numbers.length > 0 ? 
                        `<br><small class=\"text-muted\">Serial: ${item.serial_numbers.join(', ')}</small>` 
                        : ''}
                </td>
                <td style="text-align: center;">
                    <input type=\"number\" class=\"form-control form-control-sm\" 
                           value=\"${item.quantity}\" min=\"1\" 
                           onchange=\"updateQuantity(${index}, this.value)\">
                </td>
                <td style="text-align: center;">
                    <input type=\"number\" class=\"form-control form-control-sm\" 
                           value=\"${item.price}\" min=\"${(item.cost_price !== undefined ? item.cost_price : 0).toFixed(2)}\" step=\"0.01\" 
                           onchange=\"updatePrice(${index}, this.value)\"
                           oninput=\"validatePriceInput(this, ${index}, ${item.cost_price !== undefined ? item.cost_price : 0})\">
                </td>
                <td style="text-align: center;">${(item.total || (item.price * item.quantity)).toFixed(2)}</td>
                <td style="text-align: center;">
                    <input type=\"number\" class=\"form-control form-control-sm\" 
                           value=\"${item.discount ? (item.discount / item.quantity).toFixed(2) : 0}\" min=\"0\" 
                           onchange=\"updateDiscount(${index}, this.value)\">

                </td>
                <td style="text-align: center;">
                    <button class=\"btn btn-sm btn-danger\" onclick=\"removeItem(${index})\">
                        <i class=\"fas fa-trash\"></i>
                    </button>
                </td>
            `;
            cartBody.appendChild(row);
            total += (item.total || (item.price * item.quantity));
        });

        document.getElementById('cartTotal').textContent = total.toFixed(2);
        document.getElementById('summaryPayable').textContent = total.toFixed(2);
    }

    function updateSerialNumbers(index, selectedOptions) {
        const selectedSerials = Array.from(selectedOptions).map(option => option.value);
        cart[index].serial_numbers = selectedSerials;
        updateCart();
    }

    function removeItem(index) {
        console.log('removeItem called with index:', index);
        console.log('Cart before removal:', cart);
        console.log('Cart length before removal:', cart.length);
        
        if (index < 0 || index >= cart.length) {
            console.error('Invalid index:', index, 'Cart length:', cart.length);
            alert('Error: Invalid item index');
            return;
        }
        
        const removedItem = cart.splice(index, 1);
        console.log('Removed item:', removedItem);
        console.log('Cart after removal:', cart);
        console.log('Cart length after removal:', cart.length);
        
        updateCart();
    }

    function processSale() {
        if (cart.length === 0) {
            alert('Cart is empty');
            return;
        }

        const paymentMethod = document.getElementById('payment-method').value;
        const amountPaid = parseFloat(document.getElementById('amount-paid').value);
        const total = cart.reduce((sum, item) => sum + item.total, 0);

        if (amountPaid < total) {
            alert('Amount paid is less than total amount');
            return;
        }

        // Prepare sale data
        const saleData = {
            customer_id: currentCustomer ? currentCustomer.id : null,
            invoice_number: currentInvoice,
            payment_method: paymentMethod,
            amount_paid: amountPaid,
            items: cart.map(item => ({
                product_id: item.id,
                quantity: item.quantity,
                price: item.price,
                total: item.total,
                serial_numbers: item.serial_numbers.join(',')
            }))
        };

        // Send sale data to server
        fetch('api/process_sale.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(saleData)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Sale completed successfully');
                // Clear cart and reset form
                cart = [];
                updateCart();
                document.getElementById('payment-method').value = 'cash';
                document.getElementById('amount-paid').value = '';
                // Print receipt
                window.open(`receipts/sales_receipt.php?sale_id=${data.sale_id}`, '_blank');
            } else {
                alert(data.message || 'Error processing sale');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error processing sale');
        });
    }

    // Function to show serial number modal
    function showSerialNumberModal(serialNumbers, product) {
        console.log('showSerialNumberModal called with:', { serialNumbers, product });
        
        const serialNumberList = document.getElementById('serialNumberList');
        const serialQuantity = document.getElementById('serialQuantity');
        const modalTitle = document.getElementById('serialNumberModalLabel');
        
        // Update modal title with product name
        modalTitle.textContent = `Select Serial Numbers - ${product.name}`;
        
        serialNumberList.innerHTML = '';
        serialQuantity.value = 1;
        serialQuantity.max = serialNumbers.length;
        
        if (serialNumbers.length === 0) {
            serialNumberList.innerHTML = `
                <div class="text-center text-warning p-4">
                    <i class="fas fa-exclamation-triangle fa-2x mb-3"></i>
                    <p class="mb-0 fw-bold">No Serial Numbers Available</p>
                    <p class="mb-2 text-muted">This product requires serial numbers but none are currently available.</p>
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Possible reasons:</strong>
                        <ul class="mb-0 mt-2">
                            <li>All serial numbers have been sold</li>
                            <li>Serial numbers haven't been added to inventory yet</li>
                            <li>Serial numbers need to be imported or generated</li>
                        </ul>
                    </div>
                    <div class="mt-3">
                        <button type="button" class="btn btn-outline-primary btn-sm me-2" onclick="window.open('products.php', '_blank')">
                            <i class="fas fa-cog me-2"></i>Manage Products
                        </button>
                        <button type="button" class="btn btn-outline-info btn-sm" onclick="showSerialNumberHelp()">
                            <i class="fas fa-question-circle me-2"></i>Get Help
                        </button>
                    </div>
                </div>
            `;
            document.getElementById('confirmSerialSelection').disabled = true;
        } else {
            document.getElementById('confirmSerialSelection').disabled = false;
            
            // Create a grid layout for serial numbers
            const gridContainer = document.createElement('div');
            gridContainer.className = 'row g-2';
            
            serialNumbers.forEach((serial, idx) => {
                const id = `serialOption${idx}`;
                const col = document.createElement('div');
                col.className = 'col-md-6 col-lg-4';
                col.innerHTML = `
                    <div class="serial-number-item p-2 border rounded" style="background: white; transition: all 0.2s ease;">
                    <div class="form-check">
                            <input class="form-check-input serial-checkbox" type="checkbox" value="${serial}" id="${id}" 
                                   style="transform: scale(1.2); margin-top: 0.2rem;">
                            <label class="form-check-label" for="${id}" style="font-weight: 500; color: #374151; cursor: pointer;">
                                <i class="fas fa-barcode me-2" style="color: #10b981;"></i>${serial}
                            </label>
                        </div>
                    </div>
                `;
                gridContainer.appendChild(col);
            });
            
            serialNumberList.appendChild(gridContainer);
            
            // Add event listeners for better UX
            setTimeout(() => {
                const checkboxes = document.querySelectorAll('.serial-checkbox');
                checkboxes.forEach(checkbox => {
                    checkbox.addEventListener('change', function() {
                        const item = this.closest('.serial-number-item');
                        if (this.checked) {
                            item.style.background = '#ecfdf5';
                            item.style.borderColor = '#10b981';
                        } else {
                            item.style.background = 'white';
                            item.style.borderColor = '#e5e7eb';
                        }
                        updateSerialSelection();
                    });
                });
                
                // Add quantity change listener
                const quantityInput = document.getElementById('serialQuantity');
                quantityInput.addEventListener('change', function() {
                    updateSerialSelection();
                });
            }, 100);
        }
        
        // Show modal
        const modal = new bootstrap.Modal(document.getElementById('serialNumberModal'));
        modal.show();
    }

    // Function to update serial selection validation
    function updateSerialSelection() {
        const checkedBoxes = document.querySelectorAll('.serial-checkbox:checked');
        const selectedCount = checkedBoxes.length;
        const quantity = parseInt(document.getElementById('serialQuantity').value) || 1;
        const confirmBtn = document.getElementById('confirmSerialSelection');
        
        if (selectedCount === quantity) {
            confirmBtn.disabled = false;
            confirmBtn.innerHTML = '<i class="fas fa-check me-2"></i>Add to Cart';
        } else if (selectedCount > quantity) {
            confirmBtn.disabled = true;
            confirmBtn.innerHTML = `<i class="fas fa-exclamation-triangle me-2"></i>Too Many Selected (${selectedCount}/${quantity})`;
        } else {
            confirmBtn.disabled = true;
            confirmBtn.innerHTML = `<i class="fas fa-info-circle me-2"></i>Select ${quantity - selectedCount} More`;
        }
    }

    // Function to select all serial numbers
    function selectAllSerials() {
        const quantity = parseInt(document.getElementById('serialQuantity').value) || 1;
        const checkboxes = document.querySelectorAll('.serial-checkbox');
        
        // Clear all first
        checkboxes.forEach(checkbox => {
            checkbox.checked = false;
            const item = checkbox.closest('.serial-number-item');
            item.style.background = 'white';
            item.style.borderColor = '#e5e7eb';
        });
        
        // Select up to the quantity limit
        for (let i = 0; i < Math.min(quantity, checkboxes.length); i++) {
            checkboxes[i].checked = true;
            const item = checkboxes[i].closest('.serial-number-item');
            item.style.background = '#ecfdf5';
            item.style.borderColor = '#10b981';
        }
        
        updateSerialSelection();
    }

    // Function to clear all serial numbers
    function clearAllSerials() {
        const checkboxes = document.querySelectorAll('.serial-checkbox');
        checkboxes.forEach(checkbox => {
            checkbox.checked = false;
            const item = checkbox.closest('.serial-number-item');
            item.style.background = 'white';
            item.style.borderColor = '#e5e7eb';
        });
        
        updateSerialSelection();
    }

    // Function to show help for missing serial numbers
    function showSerialNumberHelp() {
        const helpModal = `
            <div class="modal fade" id="serialHelpModal" tabindex="-1">
                <div class="modal-dialog modal-lg">
                    <div class="modal-content">
                        <div class="modal-header bg-primary text-white">
                            <h5 class="modal-title">
                                <i class="fas fa-question-circle me-2"></i>Serial Number Help
                            </h5>
                            <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <div class="row">
                                <div class="col-md-6">
                                    <h6><i class="fas fa-plus-circle text-success me-2"></i>Adding Serial Numbers</h6>
                                    <ol>
                                        <li>Go to <strong>Products</strong> page</li>
                                        <li>Find your product</li>
                                        <li>Click <strong>Edit</strong></li>
                                        <li>Go to <strong>Serial Numbers</strong> tab</li>
                                        <li>Add serial numbers manually or import them</li>
                                    </ol>
                                </div>
                                <div class="col-md-6">
                                    <h6><i class="fas fa-exclamation-triangle text-warning me-2"></i>Common Issues</h6>
                                    <ul>
                                        <li><strong>No serials found:</strong> Add serial numbers to the product</li>
                                        <li><strong>All serials sold:</strong> Add more inventory</li>
                                        <li><strong>Wrong category:</strong> Check if product category requires serials</li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                            <button type="button" class="btn btn-primary" onclick="window.open('products.php', '_blank')">
                                <i class="fas fa-external-link-alt me-2"></i>Go to Products
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        // Remove existing modal if any
        const existingModal = document.getElementById('serialHelpModal');
        if (existingModal) {
            existingModal.remove();
        }
        
        // Add modal to body
        document.body.insertAdjacentHTML('beforeend', helpModal);
        
        // Show modal
        const modal = new bootstrap.Modal(document.getElementById('serialHelpModal'));
        modal.show();
    }

    // Handle serial number selection
    document.getElementById('confirmSerialSelection').addEventListener('click', function() {
        const checkedBoxes = document.querySelectorAll('.serial-checkbox:checked');
        const selectedSerials = Array.from(checkedBoxes).map(cb => cb.value);
        const quantity = parseInt(document.getElementById('serialQuantity').value) || 1;
        
        if (selectedSerials.length !== quantity) {
            showNotification(`Please select exactly ${quantity} serial number(s).`, 'warning');
            return;
        }
        
        if (pendingProduct) {
            pendingProduct.quantity = quantity;
            pendingProduct.serial_numbers = selectedSerials;
            
            // Check if product is already in cart
            const existingIndex = cart.findIndex(item => item.id === pendingProduct.id);
            if (existingIndex > -1) {
                // Update quantity and serials if stock allows
                if (cart[existingIndex].quantity + quantity <= pendingProduct.stock_quantity) {
                    cart[existingIndex].quantity += quantity;
                    cart[existingIndex].serial_numbers = cart[existingIndex].serial_numbers.concat(selectedSerials);
                } else {
                    showNotification('Cannot add more items. Maximum stock quantity reached.', 'error');
                    return;
                }
            } else {
                cart.push({
                    ...pendingProduct, 
                    cost_price: typeof pendingProduct.cost_price !== 'undefined' ? pendingProduct.cost_price : 0,
                    original_price: pendingProduct.price
                });
            }
            
            updateCartDisplay();
            
            // Hide modal
            const modal = bootstrap.Modal.getInstance(document.getElementById('serialNumberModal'));
            modal.hide();
            
            // Show success message
            showNotification(`Added ${quantity} item(s) with ${selectedSerials.length} serial number(s) to cart!`, 'success');
            
            pendingProduct = null;
        }
    });

    // Function to validate price input in real-time
    function validatePriceInput(input, index, costPrice) {
        const newPrice = parseFloat(input.value);
        const minPrice = costPrice;
        
        // Debug: Log the validation values
        console.log('Price validation:', {
            newPrice: newPrice,
            minPrice: minPrice,
            costPrice: costPrice,
            item: cart[index]
        });
        
        if (newPrice < minPrice) {
            input.style.borderColor = '#dc3545';
            input.style.backgroundColor = '#fff5f5';
        } else {
            input.style.borderColor = '';
            input.style.backgroundColor = '';
        }
    }

    // Fullscreen POS functionality
    document.addEventListener('DOMContentLoaded', function() {
        const expandBtn = document.getElementById('expandPosBtn');
        const posCard = document.querySelector('.pos-main-card');
        let isFullscreen = false;

        expandBtn.addEventListener('click', function() {
            if (!isFullscreen) {
                // Enter fullscreen
                posCard.classList.add('fullscreen');
                expandBtn.classList.add('fullscreen');
                expandBtn.innerHTML = '<i class="fas fa-compress"></i>';
                expandBtn.title = 'Exit Full Screen';
                isFullscreen = true;
                
                // Show overlay
                const overlay = document.getElementById('fullscreenOverlay');
                if (overlay) overlay.classList.add('active');
                
                // Hide sidebar and topbar
                const sidebar = document.querySelector('.sidebar');
                const topbar = document.querySelector('.topbar');
                if (sidebar) sidebar.style.display = 'none';
                if (topbar) topbar.style.display = 'none';
                
                // Add escape key listener
                document.addEventListener('keydown', handleEscapeKey);
                
                // Show notification
                showNotification('POS expanded to full screen. Press ESC to exit.', 'info');
            } else {
                // Exit fullscreen
                posCard.classList.remove('fullscreen');
                expandBtn.classList.remove('fullscreen');
                expandBtn.innerHTML = '<i class="fas fa-expand"></i>';
                expandBtn.title = 'Expand to Full Screen';
                isFullscreen = false;
                
                // Hide overlay
                const overlay = document.getElementById('fullscreenOverlay');
                if (overlay) overlay.classList.remove('active');
                
                // Show sidebar and topbar
                const sidebar = document.querySelector('.sidebar');
                const topbar = document.querySelector('.topbar');
                if (sidebar) sidebar.style.display = '';
                if (topbar) topbar.style.display = '';
                
                // Remove escape key listener
                document.removeEventListener('keydown', handleEscapeKey);
                
                // Show notification
                showNotification('POS returned to normal view.', 'info');
            }
        });

        function handleEscapeKey(event) {
            if (event.key === 'Escape' && isFullscreen) {
                expandBtn.click();
            }
        }


    });

    // QR Code Scanner functionality
    document.addEventListener('DOMContentLoaded', function() {
        // Initialize Bootstrap tooltips
        var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
        
        const cameraBtn = document.getElementById('cameraBtn');
        const searchInput = document.getElementById('searchproducts');
        
        if (cameraBtn) {
            cameraBtn.addEventListener('click', function() {
                // Check if jsQR library is loaded
                if (typeof jsQR === 'undefined') {
                    showNotification('QR scanner library not loaded. Please refresh the page and try again.', 'error');
                    return;
                }
                
                // Check camera support
                if (!checkCameraSupport()) {
                    return;
                }
                
                // Show QR scanner modal
                const modal = new bootstrap.Modal(document.getElementById('qrScannerModal'));
                modal.show();
                
                // Initialize scanner after modal is shown
                setTimeout(() => {
                    // Reset modal state
                    document.getElementById('qrLoading').style.display = 'flex';
                    document.getElementById('qrVideo').style.display = 'none';
                    document.getElementById('qrScannerError').style.display = 'none';
                    
                    // Remove any existing mobile instructions
                    const existingInstructions = document.querySelector('.mobile-instructions');
                    if (existingInstructions) {
                        existingInstructions.remove();
                    }
                    
                    // Start camera with mobile-friendly approach
                    requestCameraPermission();
                }, 500);
            });
        }
        
        // Clean up when QR scanner modal is closed
        const qrScannerModal = document.getElementById('qrScannerModal');
        if (qrScannerModal) {
            qrScannerModal.addEventListener('hidden.bs.modal', function() {
                stopQRScanner();
            });
        }
    });

    // QR Scanner variables
    let videoStream = null;
    let qrScanner = null;

    // Global notification function
        function showNotification(message, type = 'info') {
            // Create notification element
            const notification = document.createElement('div');
            notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
            notification.style.cssText = 'top: 20px; right: 20px; z-index: 10000; min-width: 300px; max-width: 400px;';
            notification.innerHTML = `
                <div class="d-flex align-items-center">
                <i class="fas fa-${type === 'info' ? 'info-circle' : type === 'success' ? 'check-circle' : type === 'warning' ? 'exclamation-triangle' : 'exclamation-circle'} me-2"></i>
                    <span>${message}</span>
                    <button type="button" class="btn-close ms-auto" data-bs-dismiss="alert"></button>
                </div>
            `;
            
            // Add to body
            document.body.appendChild(notification);
            
            // Auto remove after 3 seconds
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.remove();
                }
            }, 3000);
        }

    function initializeQRScanner() {
        const video = document.getElementById('qrVideo');
        const canvas = document.getElementById('qrCanvas');
        const ctx = canvas.getContext('2d');
        
        // Check if mediaDevices is supported
        if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
            showCameraError('Camera API not supported in this browser. Please use a modern browser.');
            return;
        }
        
        // Try different camera configurations for mobile devices
        const cameraConfigs = [
            // Mobile-friendly configuration
            {
                video: {
                    facingMode: { ideal: 'environment' },
                    width: { min: 320, ideal: 640, max: 1280 },
                    height: { min: 240, ideal: 480, max: 720 },
                    aspectRatio: { ideal: 4/3 }
                }
            },
            // Fallback configuration
            {
                video: {
                    facingMode: 'environment',
                    width: { ideal: 640 },
                    height: { ideal: 480 }
                }
            },
            // Basic configuration
            {
                video: true
            }
        ];
        
        async function tryCameraConfig(configIndex) {
            if (configIndex >= cameraConfigs.length) {
                showCameraError('Unable to access camera. Please check permissions and try again.');
                return;
            }
            
            try {
                console.log(`Trying camera config ${configIndex + 1}:`, cameraConfigs[configIndex]);
                
                const stream = await navigator.mediaDevices.getUserMedia(cameraConfigs[configIndex]);
                videoStream = stream;
                video.srcObject = stream;
                
                // Hide loading and show video
                document.getElementById('qrLoading').style.display = 'none';
                video.style.display = 'block';
                
                // Wait for video to be ready
                video.onloadedmetadata = function() {
                    video.play();
                    // Start scanning
                    scanQRCode();
                };
                
            } catch (err) {
                console.error(`Camera config ${configIndex + 1} failed:`, err);
                
                // Try next configuration
                if (err.name === 'NotAllowedError') {
                    showCameraError('Camera access denied. Please allow camera permissions in your browser settings and try again.');
                } else if (err.name === 'NotFoundError') {
                    showCameraError('No camera found on this device. Please connect a camera and try again.');
                } else if (err.name === 'NotSupportedError') {
                    showCameraError('Camera not supported on this device/browser. Please try a different browser.');
                } else {
                    // Try next configuration
                    setTimeout(() => tryCameraConfig(configIndex + 1), 500);
                }
            }
        }
        
        function showCameraError(message) {
            const errorElement = document.getElementById('qrScannerError');
            const errorTextElement = document.getElementById('qrScannerErrorText');
            
            // Hide loading
            document.getElementById('qrLoading').style.display = 'none';
            
            errorTextElement.textContent = message;
            errorElement.style.display = 'block';
        }
        
        // Start with first configuration
        tryCameraConfig(0);
    }

    function scanQRCode() {
        const video = document.getElementById('qrVideo');
        const canvas = document.getElementById('qrCanvas');
        const ctx = canvas.getContext('2d');
        
        if (video.readyState === video.HAVE_ENOUGH_DATA) {
            canvas.height = video.videoHeight;
            canvas.width = video.videoWidth;
            ctx.drawImage(video, 0, 0, canvas.width, canvas.height);
            
            const imageData = ctx.getImageData(0, 0, canvas.width, canvas.height);
            
            // Use jsQR library to detect QR codes
            const code = jsQR(imageData.data, imageData.width, imageData.height, {
                inversionAttempts: "dontInvert",
            });
            
            if (code) {
                // QR code detected
                console.log('QR Code detected:', code.data);
                
                // Stop scanning
                stopQRScanner();
                
                // Set the scanned value to search input
                document.getElementById('searchproducts').value = code.data;
                
                // Close QR scanner modal
                const modal = bootstrap.Modal.getInstance(document.getElementById('qrScannerModal'));
                modal.hide();
                
                // Show success message
                showNotification('QR Code scanned successfully!', 'success');
                
                // Play success sound if available
                try {
                    const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBSuBzvLZiTYIG2m98OScTgwOUarm7blmGgU7k9n1unEiBC13yO/eizEIHWq+8+OWT');
                    audio.play();
                } catch (e) {
                    // Ignore audio errors
                }
                
                // Process the scanned data as a barcode
                setTimeout(() => {
                    handleBarcodeScan(code.data);
                }, 500);
            } else {
                // Continue scanning
                requestAnimationFrame(scanQRCode);
            }
        } else {
            // Video not ready yet, try again
            requestAnimationFrame(scanQRCode);
        }
    }

    function stopQRScanner() {
        if (videoStream) {
            videoStream.getTracks().forEach(track => track.stop());
            videoStream = null;
        }
        if (qrScanner) {
            qrScanner = null;
        }
    }





    // Manual QR code input function
    function showManualQRInput() {
        const manualInput = prompt('Enter QR code data manually:');
        if (manualInput && manualInput.trim() !== '') {
            const trimmedInput = manualInput.trim();
            console.log('Manual QR input:', trimmedInput);
            
            // Set the scanned value to search input
            document.getElementById('searchproducts').value = trimmedInput;
            
            // Close QR scanner modal
            const modal = bootstrap.Modal.getInstance(document.getElementById('qrScannerModal'));
            modal.hide();
            
            // Show success message
            showNotification('QR code data entered successfully!', 'success');
            
            // Clear any existing product suggestions
            const productSuggestionsContainer = document.getElementById('productSuggestionsContainer');
            if (productSuggestionsContainer) {
                productSuggestionsContainer.style.display = 'none';
            }
            
            // Process the scanned data as a barcode with a longer delay to ensure modal is closed
            setTimeout(() => {
                console.log('Processing manual input as barcode:', trimmedInput);
                handleBarcodeScan(trimmedInput);
            }, 1000);
        }
    }
    
    // Mobile-specific camera permission request
    function requestCameraPermission() {
        // Check if we're on a mobile device
        const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
        
        if (isMobile) {
            // Show mobile-specific instructions
            const instructions = `
                <div class="alert alert-info">
                    <h6><i class="fas fa-mobile-alt"></i> Mobile Camera Setup</h6>
                    <ol class="mb-0">
                        <li>Tap "Allow" when prompted for camera access</li>
                        <li>If denied, go to browser settings and enable camera</li>
                        <li>Ensure you're using HTTPS (secure connection)</li>
                        <li>Try refreshing the page if camera doesn't work</li>
                    </ol>
                </div>
            `;
            
            // Show instructions before starting camera
            const modalBody = document.querySelector('#qrScannerModal .modal-body');
            const existingInstructions = modalBody.querySelector('.mobile-instructions');
            if (!existingInstructions) {
                const instructionsDiv = document.createElement('div');
                instructionsDiv.className = 'mobile-instructions mb-3';
                instructionsDiv.innerHTML = instructions;
                modalBody.insertBefore(instructionsDiv, modalBody.firstChild);
            }
        }
        
        // Start camera initialization
        initializeQRScanner();
    }
    
    // Check if device supports camera
    function checkCameraSupport() {
        const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
        const isSecure = window.location.protocol === 'https:' || window.location.hostname === 'localhost';
        
        if (!isSecure && !isMobile) {
            showNotification('Camera requires HTTPS connection for security', 'warning');
        }
        
        if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
            showNotification('Camera not supported in this browser', 'error');
            return false;
        }
        
        // Check for permissions policy violation
        if (typeof navigator.permissions !== 'undefined') {
            navigator.permissions.query({ name: 'camera' }).then(function(result) {
                if (result.state === 'denied') {
                    showNotification('Camera access is blocked by browser. Please enable camera permissions in browser settings.', 'error');
                    return false;
                }
            }).catch(function(err) {
                console.log('Permissions API not supported:', err);
            });
        }
        
        return true;
    }
    
    // Handle mobile-specific camera issues
    function handleMobileCameraIssues() {
        const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
        
        if (isMobile) {
            // Add mobile-specific error handling
            window.addEventListener('error', function(e) {
                if (e.message.includes('Permissions policy violation') || e.message.includes('camera is not allowed')) {
                    showNotification('Camera access blocked. Please check browser settings and allow camera permissions.', 'error');
                }
            });
            
            // Handle unhandled promise rejections
            window.addEventListener('unhandledrejection', function(e) {
                if (e.reason && e.reason.name === 'NotAllowedError') {
                    showNotification('Camera access denied. Please allow camera permissions in your browser settings.', 'error');
                }
            });
        }
    }
    
    // Initialize mobile camera handling
    handleMobileCameraIssues();
</script>

<!-- QR Scanner Modal -->
<div class="modal fade" id="qrScannerModal" tabindex="-1" aria-labelledby="qrScannerModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered modal-lg">
        <div class="modal-content" style="border-radius: 20px; border: none; box-shadow: 0 20px 60px rgba(0,0,0,0.15);">
            <div class="modal-header" style="background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%); color: white; border-radius: 20px 20px 0 0; border: none; padding: 1.5rem;">
                <div class="d-flex align-items-center">
                    <div class="me-3" style="width: 50px; height: 50px; background: rgba(255,255,255,0.2); border-radius: 50%; display: flex; align-items: center; justify-content: center;">
                        <i class="fas fa-qrcode" style="font-size: 1.5rem;"></i>
                    </div>
                    <div>
                        <h5 class="modal-title mb-0" id="qrScannerModalLabel" style="font-weight: 700; font-size: 1.3rem;">QR Code Scanner</h5>
                        <small style="opacity: 0.9;">Point camera at QR code to scan</small>
                    </div>
                </div>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" style="padding: 2rem;">
                                    <div class="text-center">
                        <div class="qr-scanner-container" style="position: relative; display: inline-block; border-radius: 15px; overflow: hidden; box-shadow: 0 10px 30px rgba(0,0,0,0.2);">
                            <div id="qrLoading" class="text-center p-4" style="background: #000; color: white; min-height: 300px; display: flex; align-items: center; justify-content: center;">
                                <div>
                                    <div class="spinner-border text-primary mb-3" role="status">
                                        <span class="visually-hidden">Loading...</span>
                                    </div>
                                    <p class="mb-0">Initializing camera...</p>
                                </div>
                            </div>
                            <video id="qrVideo" style="width: 100%; max-width: 500px; height: auto; border-radius: 15px; display: none;" autoplay muted playsinline></video>
                            <canvas id="qrCanvas" style="display: none;"></canvas>
                        
                        <!-- Scanner overlay -->
                        <div class="scanner-overlay" style="position: absolute; top: 0; left: 0; right: 0; bottom: 0; pointer-events: none;">
                            <div class="scanner-frame" style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); width: 200px; height: 200px; border: 3px solid #2563eb; border-radius: 15px; box-shadow: 0 0 0 9999px rgba(0,0,0,0.5);">
                                <div class="scanner-corner" style="position: absolute; top: -3px; left: -3px; width: 20px; height: 20px; border-top: 4px solid #2563eb; border-left: 4px solid #2563eb; border-radius: 5px 0 0 0;"></div>
                                <div class="scanner-corner" style="position: absolute; top: -3px; right: -3px; width: 20px; height: 20px; border-top: 4px solid #2563eb; border-right: 4px solid #2563eb; border-radius: 0 5px 0 0;"></div>
                                <div class="scanner-corner" style="position: absolute; bottom: -3px; left: -3px; width: 20px; height: 20px; border-bottom: 4px solid #2563eb; border-left: 4px solid #2563eb; border-radius: 0 0 0 5px;"></div>
                                <div class="scanner-corner" style="position: absolute; bottom: -3px; right: -3px; width: 20px; height: 20px; border-bottom: 4px solid #2563eb; border-right: 4px solid #2563eb; border-radius: 0 0 5px 0;"></div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="qrScannerError" class="alert alert-danger mt-3" style="display:none; border-radius: 10px; border: none; background: #fef2f2; color: #dc2626;">
                        <i class="fas fa-exclamation-circle me-2"></i>
                        <span id="qrScannerErrorText"></span>
                    </div>
                    
                    <div class="mt-3">
                        <p class="text-muted mb-0">
                            <i class="fas fa-info-circle me-2"></i>
                            Position the QR code within the scanning frame
                        </p>
                        <div class="mt-3">
                            <button type="button" class="btn btn-outline-secondary btn-sm" onclick="showManualQRInput()">
                                <i class="fas fa-keyboard me-2"></i>Manual Input
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer" style="border: none; padding: 1.5rem 2rem 2rem 2rem; background: #f9fafb;">
                <button type="button" class="btn btn-light" data-bs-dismiss="modal" 
                        style="border-radius: 10px; padding: 0.75rem 1.5rem; font-weight: 600; border: 2px solid #e5e7eb;">
                    <i class="fas fa-times me-2"></i>Cancel
                </button>
                <button type="button" class="btn btn-primary" onclick="requestCameraPermission()" 
                        style="border-radius: 10px; padding: 0.75rem 1.5rem; font-weight: 600; background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%); border: none;">
                    <i class="fas fa-camera me-2"></i>Start Camera
                </button>
                <button type="button" class="btn btn-outline-secondary" onclick="showManualQRInput()" 
                        style="border-radius: 10px; padding: 0.75rem 1.5rem; font-weight: 600;">
                    <i class="fas fa-keyboard me-2"></i>Manual Input
                </button>
            </div>
        </div>
    </div>
</div>

<?php include '../includes/footer.php'; ?>
</body>
</html>
